CREATE OR REPLACE PACKAGE Sim2_Import_Station_Pkg IS
  -- Author  : KHENAS,DanL
  -- Created : 11/15/00 11:35:05 AM
  -- Purpose : import stations data into SIM tables from a text file
   TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;
   TYPE col_table_num_type IS TABLE OF NUMBER(3) INDEX BY BINARY_INTEGER;
   TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;
   TYPE t_storet_pk_type IS RECORD (
     is_number         NUMBER(8),
     org_id            VARCHAR2(8)
     );
   TYPE t_storet_tsmvstc_type IS RECORD (
     tsmvstc_is_number         NUMBER(8),
     tsmvstc_org_id            VARCHAR2(8)
     );
    TYPE t_storet_tsmmad_type IS RECORD (
     tsmmad_is_number         NUMBER(8),
     tsmmad_org_id            VARCHAR2(8)
     );
   TYPE t_aquifer_type    IS RECORD (
     tsmgeoun_cd          VARCHAR2(12),
     aquifer_name         VARCHAR2(80)
     );
   v_activity_count     t_activity_count;
   v_org_is_nbr         NUMBER(8) := 0;
   v_import_log_seq     NUMBER(10);
   v_status             VARCHAR2(1);
   v_line_cnt           NUMBER(10) := 0;
  -- Public function and procedure declarations
  FUNCTION CHECK_IF_STATION_EXISTS(p_station_cd IN VARCHAR2,p_org_id IN VARCHAR2) RETURN NUMBER;
  FUNCTION validate_rf1(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER) RETURN VARCHAR2;
   FUNCTION validate_rf1mile(p_orig_item IN VARCHAR2, p_is_number IN NUMBER, p_org_id IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                            p_item_name IN VARCHAR2, p_col_seq IN NUMBER, p_sta_seq IN NUMBER) RETURN VARCHAR2;
   FUNCTION get_org_info(p_imp_cfg_seq IN NUMBER, p_tsmorgan_org_id IN VARCHAR2,p_tsmorgan_is_nbr OUT NUMBER) RETURN NUMBER;
   FUNCTION get_delimiter(p_imp_cfg_seq IN NUMBER,p_delimiter IN OUT VARCHAR2) RETURN NUMBER;
   FUNCTION get_item_pos(p_sicc_seq IN NUMBER, p_pos_table IN OUT col_table_num_type,
                         p_default_table IN OUT col_table_type,p_col_cnt IN OUT NUMBER) RETURN NUMBER;
   FUNCTION get_column_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) RETURN NUMBER;
   FUNCTION check_translations(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,
                               p_trans_exist OUT BOOLEAN) RETURN VARCHAR2;
   PROCEDURE create_trans_record(p_code IN VARCHAR2,p_imp_cfg_seq IN NUMBER,p_col_seq IN NUMBER,p_sta_seq IN NUMBER);
   FUNCTION check_allowable_values(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,p_sta_seq IN NUMBER)
     RETURN NUMBER;
   FUNCTION write_error_message(p_item_name IN VARCHAR2,p_req IN BOOLEAN,p_line_nbr IN NUMBER,
                                p_provided IN BOOLEAN,p_status IN OUT VARCHAR2, p_comment IN OUT VARCHAR2,
                                p_sta_seq IN NUMBER) RETURN NUMBER;
   FUNCTION get_time(p_raw_time IN VARCHAR2,p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) RETURN DATE;
  FUNCTION get_tsmgeopa_is_nbr (p_state IN VARCHAR2,p_county IN VARCHAR2) RETURN T_STORET_PK_TYPE;
  FUNCTION get_station_is_nbr (p_station_id IN VARCHAR2, p_org_id IN VARCHAR2,p_item_name IN VARCHAR2, p_sta_seq IN NUMBER)
    RETURN T_STORET_PK_TYPE;
  FUNCTION get_tsmfhu_is_nbr (p_fhu IN VARCHAR2) RETURN T_STORET_PK_TYPE;
  FUNCTION get_tsmrrr_is_nbr (p_huc IN VARCHAR2, p_rf1 IN VARCHAR2) RETURN T_STORET_PK_TYPE;
  PROCEDURE import_station (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                          p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER);
END Sim2_Import_Station_Pkg;
/
CREATE OR REPLACE PACKAGE BODY Sim2_Import_Station_Pkg IS
  e_a   EXCEPTION;
  -- Function and procedure implementations
FUNCTION CHECK_IMPORT_TYPE(p_cfg_seq IN NUMBER) RETURN NUMBER IS
  v_import_type     VARCHAR2(30);
  CURSOR c_get_imp_type(p_imp_cfg_seq IN NUMBER) IS
    SELECT SID_TYPE FROM SIM_IMP_DEFS, SIM_IMP_CFG
      WHERE SICC_SID_SEQ = SID_SEQ AND SICC_SEQ = p_imp_cfg_seq;
BEGIN
 -- v_org_id := NAME_IN('IMPSTAT.L_SICG_SIC_ORG_ID');
  OPEN c_get_imp_type(p_cfg_seq);
  FETCH c_get_imp_type INTO v_import_type;
  CLOSE c_get_imp_type;
  IF v_import_type = 'GW STATIONS' THEN
   RETURN 1;
  ELSE
   RETURN 0;
  END IF;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END CHECK_IMPORT_TYPE;
/*********************************************************************************************************/
  -- Function and procedure implementations
/** This function checks to see if a station already exists by checking for existing station ID**/
FUNCTION CHECK_IF_STATION_EXISTS(p_station_cd IN VARCHAR2,p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_station_cd     VARCHAR2(15);
  
  CURSOR c_compare_station_id(p_station_cd IN VARCHAR2,p_station_org_id IN VARCHAR2) IS
  SELECT IDENTIFICATION_CD FROM TSMSTATN
  WHERE  IDENTIFICATION_CD = RPAD(p_station_cd,15)
  AND TSMSTATN_ORG_ID = RPAD(p_org_id,8);
  
  CURSOR c_compare_sim_stat_id(p_station_cd IN VARCHAR2,p_station_org_id IN VARCHAR2) IS
  SELECT STA_ID FROM SIM_STATIONS
  WHERE  STA_ID = p_station_cd
  AND STA_TSMORGAN_ORG_ID = p_org_id;
  
BEGIN
--  sim2_import_pkg.write_error(v_import_log_seq,'V_STATION_CD value: '||v_station_cd,null,null,null);

  /* Check STORET */
  OPEN c_compare_station_id(p_station_cd,p_org_id);
  FETCH c_compare_station_id INTO v_station_cd;
  CLOSE c_compare_station_id;
  IF v_station_cd IS NOT NULL THEN  
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Station with this Station ID already exists in STORET - station will not export '||v_station_cd||' line: '||v_line_cnt,NULL,NULL,NULL);
  END IF;
  
  /* Check SIM */
  IF v_station_cd IS NULL THEN
    OPEN c_compare_sim_stat_id(p_station_cd,p_org_id);
    FETCH c_compare_sim_stat_id INTO v_station_cd;
    CLOSE c_compare_sim_stat_id;
    IF v_station_cd IS NOT NULL THEN  
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Station with this Station ID already exists in SIM - station will not export '||v_station_cd||' line: '||v_line_cnt,NULL,NULL,NULL);
    END IF;
  END IF;
  
  /** Station is unique **/
  IF v_station_cd IS NULL THEN
    RETURN 1;
  ELSE
    RETURN 0;
  END IF;
  
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check_if_station_exists.line: '||v_line_cnt,NULL,NULL,NULL);
    RETURN 0;
END CHECK_IF_STATION_EXISTS;
/*********************************************************************************************************/
FUNCTION get_org_info(p_imp_cfg_seq IN NUMBER, p_tsmorgan_org_id IN VARCHAR2,p_tsmorgan_is_nbr OUT NUMBER) RETURN NUMBER IS
  CURSOR c_get_tsmorgan_nbr (p_cursor_org_id IN VARCHAR2) IS
    SELECT TSMORGAN_IS_NUMBER FROM TSMORGAN WHERE ORG_ID = RPAD(p_cursor_org_id,8);
BEGIN
  OPEN c_get_tsmorgan_nbr (p_tsmorgan_org_id);
  FETCH c_get_tsmorgan_nbr INTO p_tsmorgan_is_nbr;
    IF c_get_tsmorgan_nbr%NOTFOUND THEN
      p_tsmorgan_is_nbr := NULL;
    END IF;
  CLOSE c_get_tsmorgan_nbr;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20199 , 'Failed to get organization is number.');
    RETURN 0;
END get_org_info;
/******************************************************************************************************/
FUNCTION get_instructions(p_imp_cfg_seq IN NUMBER, p_instruction_type IN VARCHAR2, p_fa_seq IN NUMBER)
  RETURN VARCHAR2 IS
  v_instr_choice         VARCHAR2(15);
  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SICI_SICC_SEQ = p_cfg_seq;
  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_INSTR_CHOICES, SIM_IMP_CFG_INSTRS, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SIIC_DFLT_FLAG = 1;
BEGIN
  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr INTO v_instr_choice;
    IF c_get_cfg_instr%NOTFOUND THEN
      /* no choice could be found, get default choice */
      OPEN c_get_instr_default(p_instruction_type);
      FETCH c_get_instr_default INTO v_instr_choice;
      CLOSE c_get_instr_default;
    END IF;
  CLOSE c_get_cfg_instr;
  RETURN v_instr_choice;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get_instructions.',NULL,p_fa_seq,NULL);
    RETURN NULL;
END get_instructions;
/******************************************************************************************************/
FUNCTION get_allowable_value(p_alval_seq IN NUMBER, p_sta_seq IN NUMBER, p_item_name IN VARCHAR2) RETURN VARCHAR2 IS
  v_item_name         VARCHAR2(80);
  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_alval_seq;
BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name INTO v_item_name;
    IF c_get_alval_name%NOTFOUND THEN
      v_item_name := NULL;
    END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - getting allowable value sequence.',p_sta_seq,NULL,NULL);
END get_allowable_value;
/******************************************************************************************************
    The validate_primary_type function takes the original item and
    checks it against existing translations. If there is a translation the translation is returned
    to IMPORT_STATIONS where it is used to find the tsmvstc_is_nbr and org_id.
/****************************************************************************************************/
/******************************************************************************************************
    The validate_primary_type function takes the original item and
    checks it against existing translations. If there is a translation the translation is returned
    to IMPORT_STATIONS where it is used to find the tsmvstc_is_nbr and org_id.
/****************************************************************************************************/
FUNCTION validate_primary_type(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER,
                              p_item_name IN VARCHAR2,
                              p_station_type IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_primary_type              VARCHAR2(30);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_choice_id                 VARCHAR2(15);
CURSOR c_check_primary_type  (p_primary_type IN VARCHAR2)
       IS SELECT PRIMARY_TYPE_CD FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_primary_type;
BEGIN
  /** function to look in the translations and STORET tables for items that should already exist in
  /** hard coded values in STORET tables **/
  /** look in translations table to see if it needs to be swapped **/
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
  OPEN c_check_primary_type (v_converted_item);
   FETCH c_check_primary_type INTO v_primary_type;
    IF c_check_primary_type%NOTFOUND THEN
     v_primary_type := NULL;
    END IF;
  CLOSE c_check_primary_type;
/* call function to look in storet table for primary type */
   IF v_primary_type IS NULL AND NOT v_trans_exist THEN
       IF p_station_type = 1 THEN -- it's a groundwater station
       v_instr_id  := 'GWS-TR';
       v_choice_id := 'GWS-TR-01';
    ELSE
       v_instr_id  := 'STA-TR';
       v_choice_id := 'STA-TR-01';
    END IF;
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id, p_sta_seq);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
           create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_primary_type IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item||' line: '||V_line_cnt,NULL,p_sta_seq,NULL);
      v_status := 'N';
    END IF;
  ELSIF v_trans_exist THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_orig_item||' exists in translation table but has not been mapped to a STORET code. line: '||V_line_cnt,NULL,p_sta_seq,NULL);
    v_status := 'N';
  END IF;
  RETURN v_primary_type;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate primary type.line: '||v_line_cnt,p_sta_seq,NULL,NULL);
    RETURN NULL;
    v_status := 'N';
END validate_primary_type;
/******************************************************************************************************
    The validate_secondary_type function takes the original item and
    checks it against existing translations. If there is a translation the translation is returned
    to IMPORT_STATIONS where it is used to find the tsmvstc_is_nbr and org_id .
    If there is a valid secondary type, the is number and org id from the primary type calculation
    will be overwritten.
/****************************************************************************************************/
FUNCTION validate_secondary_type(p_orig_item IN VARCHAR2, p_primary_type IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_station_type IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_secondary_count           NUMBER(2);
  v_secondary_type            VARCHAR2(30);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_choice_id                 VARCHAR2(15);
CURSOR c_secondary_needed (p_primary_type IN VARCHAR2)
       IS  SELECT COUNT(primary_type_cd) FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_primary_type;
CURSOR c_check_secondary_type  (p_secondary_type IN VARCHAR2,p_primary_type IN VARCHAR2)
       IS SELECT SECONDARY_TYPE_CD FROM TSMVSTC
       WHERE SECONDARY_TYPE_CD = p_secondary_type AND PRIMARY_TYPE_CD = p_primary_type;
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
 IF v_converted_item IS NOT NULL THEN
/* The cursor will check if the primary type has secondary types */
   OPEN c_secondary_needed (p_primary_type);
   FETCH c_secondary_needed INTO v_secondary_count;
/*If a secondary type is permitted then find the correct secondary type */
  IF v_secondary_count > 1 THEN
   OPEN c_check_secondary_type (v_converted_item,p_primary_type);
   FETCH c_check_secondary_type INTO v_secondary_type;
    IF c_check_secondary_type%NOTFOUND THEN
     v_secondary_type := NULL;
     Sim2_Import_Pkg.write_error(v_import_log_seq,'Secondary item: '||p_orig_item||' Does not match with primary item: '||p_primary_type||' line: '||v_line_cnt,NULL,p_sta_seq,NULL);
    END IF;
   CLOSE c_check_secondary_type;
  ELSE
/* The primary type for this station cannot take a secondary type */
    v_secondary_type := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_primary_type||': This primary type -> '||p_primary_type||' cannot accept a secondary type, line: '||v_line_cnt,NULL,p_sta_seq,NULL);
    RETURN v_secondary_type;
  END IF;
/* call function to look in storet table for primary/secondary type */
    IF v_secondary_type IS NULL AND NOT v_trans_exist THEN
           IF p_station_type = 1 THEN -- it's a groundwater station
       v_instr_id  := 'GWS-TR';
       v_choice_id := 'GWS-TR-01';
    ELSE
       v_instr_id  := 'STA-TR';
       v_choice_id := 'STA-TR-01';
    END IF;
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id, p_sta_seq);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_secondary_type IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item||' line: '||v_line_cnt,NULL,p_sta_seq,NULL);
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_secondary_type := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,' Secondary type: '||p_orig_item||' - Exists in translation table but has not been mapped to a STORET code. line: '||v_line_cnt,NULL,p_sta_seq,NULL);
  END IF;
  RETURN v_secondary_type;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate secondary type. line: '||v_line_cnt,p_sta_seq,NULL,NULL);
    RETURN NULL;
END validate_secondary_type;
/******************************************************************************************************
    The validate_county function takes the original item and checks it against existing translations.
    cursor c_check_county  uses character data to search, while c_check_county_fips checks with the
    county FIPS code. First the p_orig_item is tested to see if it is of character or numeric format,
    then it is checked against known translations.
    If there is a translation the translation is returned and used in a cursor to find the IS number
    org id from TSMGEOPA.
/****************************************************************************************************/
FUNCTION validate_county(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER,
                              p_station_type IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            VARCHAR2(25);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_county                    VARCHAR2(25);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_choice_id                 VARCHAR2(15);
CURSOR c_check_county  (p_county IN VARCHAR2)
       IS SELECT COUNTY_NAME FROM TSMGEOPA
       WHERE COUNTY_NAME = p_county;
CURSOR c_check_county_fips  (p_county IN VARCHAR2)
       IS SELECT FIPS_COUNTY_CODE FROM TSMGEOPA
       WHERE FIPS_COUNTY_CODE = p_county;
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
  IF TRANSLATE(v_converted_item, 'a1234567890.- ', 'a') IS NULL THEN -- it's a FIPS county code
   OPEN c_check_county_fips (v_converted_item);
   FETCH c_check_county_fips INTO v_county;
    IF c_check_county_fips%NOTFOUND THEN
     v_county := NULL;
     Sim2_Import_Pkg.write_error(v_import_log_seq, 'FIPS county code is invalid.',p_sta_seq,NULL,NULL);
    END IF;
   CLOSE c_check_county_fips;
 ELSE -- It is a text code
    OPEN c_check_county (v_converted_item);
    FETCH c_check_county INTO v_county;
     IF c_check_county%NOTFOUND THEN
      v_county := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'County name is invalid.',p_sta_seq,NULL,NULL);
     END IF;
    CLOSE c_check_county;
 END IF;
/* call function to look in storet table for primary/secondary type */
    IF v_county IS NULL AND NOT v_trans_exist THEN
    IF p_station_type = 1 THEN -- it's a groundwater station
       v_instr_id  := 'GWS-TR';
       v_choice_id := 'GWS-TR-01';
    ELSE
       v_instr_id  := 'STA-TR';
       v_choice_id := 'STA-TR-01';
    END IF;
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id, p_sta_seq);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_county IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq,NULL,NULL);
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_county := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_orig_item||' county name exists in translation table but has not been mapped to a STORET code. ',p_sta_seq,NULL,NULL);
  END IF;
  RETURN v_county;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate county name.',p_sta_seq,NULL,NULL);
    RETURN NULL;
END validate_county;
/******************************************************************************************************
    The validate_state function takes the original item and checks it against existing translations.
    If there is a translation the translation is returned and used in a cursor to find the IS number
    org id from TSMGEOPA.
/****************************************************************************************************/
FUNCTION validate_state(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER,
                              p_station_type IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_state                     VARCHAR2(2);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_choice_id                 VARCHAR2(15);
CURSOR c_check_state  (p_state IN VARCHAR2)
       IS SELECT STATE_POSTAL_CODE FROM TSMGEOPA
       WHERE STATE_POSTAL_CODE = RPAD(p_state,2);
CURSOR c_check_state_fips  (p_state IN VARCHAR2)
       IS SELECT STATE_FIPS_CODE FROM TSMGEOPA
       WHERE STATE_FIPS_CODE = RPAD(p_state,2);
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
    IF TRANSLATE(v_converted_item, 'a1234567890.- ', 'a') IS NULL THEN -- it's a FIPS county code
     OPEN c_check_state_fips (v_converted_item);
     FETCH c_check_state_fips INTO v_state;
      IF c_check_state_fips%NOTFOUND THEN
       v_state := NULL;
      END IF;
     CLOSE c_check_state_fips;
    ELSE
     OPEN c_check_state (v_converted_item);
     FETCH c_check_state INTO v_state;
      IF c_check_state%NOTFOUND THEN
       v_state := NULL;
      END IF;
     CLOSE c_check_state;
    END IF;
/* call function to look in storet table for primary/secondary type */
    IF v_state IS NULL AND NOT v_trans_exist THEN
    IF p_station_type = 1 THEN -- it's a groundwater station
       v_instr_id  := 'GWS-TR';
       v_choice_id := 'GWS-TR-01';
    ELSE
       v_instr_id  := 'STA-TR';
       v_choice_id := 'STA-TR-01';
    END IF;
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id, p_sta_seq);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_state IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_sta_seq,NULL);
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_state := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_orig_item||' state is in translation table but has not been mapped to a STORET code. ',NULL,p_sta_seq,NULL);
  END IF;
  RETURN v_state;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate state.',p_sta_seq,NULL,NULL);
    RETURN NULL;
END validate_state;
/******************************************************************************************************
    The validate_huc function takes the original item and checks it against existing translations.
    If there is a translation the translation is returned and used in a cursor to find the IS number
    org id from TSMFHU.
/****************************************************************************************************/
FUNCTION validate_huc(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER,
                              p_station_type IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            VARCHAR2(8);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_huc                       VARCHAR2(8);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_choice_id                 VARCHAR2(15);
CURSOR c_check_huc  (p_huc IN VARCHAR2)
       IS SELECT HYDROLOGIC_UNIT_CD FROM TSMFHU
       WHERE HYDROLOGIC_UNIT_CD = RPAD(p_huc,8);
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
   OPEN c_check_huc (v_converted_item);
   FETCH c_check_huc INTO v_huc;
    IF c_check_huc%NOTFOUND THEN
     v_huc := NULL;
    END IF;
   CLOSE c_check_huc;
    IF v_huc IS NULL AND NOT v_trans_exist THEN
     IF p_station_type = 1 THEN -- it's a groundwater station
       v_instr_id  := 'GWS-TR';
       v_choice_id := 'GWS-TR-01';
     ELSE
       v_instr_id  := 'STA-TR';
       v_choice_id := 'STA-TR-01';
     END IF;
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id, p_sta_seq);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_huc IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_sta_seq,NULL);
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_huc := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_orig_item||' HUC exists in translation table but has not been mapped to a STORET code. ',NULL,p_sta_seq,NULL);
  END IF;
--DBMS_OUTPUT.PUT_LINE('at the end of val storet');
  RETURN v_huc;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate huc.',p_sta_seq,NULL,NULL);
    RETURN NULL;
END validate_huc;
/******************************************************************************************************
    The validate_rf1 function takes the original item and checks it against existing translations.
    If there is a translation the translation is returned and used in a cursor to find the IS number
    org id from TSMRRR.
/****************************************************************************************************/
FUNCTION validate_rf1(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            VARCHAR2(3);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_rf1                       VARCHAR2(3);
  v_instr_code                VARCHAR2(15);
CURSOR c_check_rf1  (p_rf1 IN VARCHAR2)
       IS SELECT SEGMENT_CODE FROM TSMRRR
       WHERE SEGMENT_CODE = p_rf1;
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
   OPEN c_check_rf1 (v_converted_item);
   FETCH c_check_rf1 INTO v_rf1;
    IF c_check_rf1%NOTFOUND THEN
     v_rf1 := NULL;
    END IF;
   CLOSE c_check_rf1;
/* call function to look in storet table for primary/secondary type */
    IF v_rf1 IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'STA-TR', p_sta_seq);
      IF v_instr_code = 'STA-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_rf1 IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_sta_seq,NULL);
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_rf1 := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_orig_item||' rf1 exists in translation table but has not been mapped to a STORET code. ',NULL,p_sta_seq,NULL);
  END IF;
  RETURN v_rf1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate rf1',p_sta_seq,NULL,NULL);
    RETURN NULL;
END validate_rf1;
/******************************************************************************************************
    The validate_rf1mile function takes the original item and checks it against existing translations.
    If there is a translation the translation is returned and used in a cursor to find the IS number
    org id from TSMRRR.
/****************************************************************************************************/
FUNCTION validate_rf1mile(p_orig_item IN VARCHAR2, p_is_number IN NUMBER, p_org_id IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                           p_item_name IN VARCHAR2, p_col_seq IN NUMBER, p_sta_seq IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            NUMBER(5,1);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_rf1mile                   NUMBER(5,1);
  v_instr_code                VARCHAR2(15);
CURSOR c_check_rf1mile  (p_is_number IN NUMBER, p_org_id IN VARCHAR2)
       IS SELECT MILE_LENGTH FROM TSMRRR
       WHERE TSMRRR_ORG_ID = p_org_id AND TSMRRR_IS_NUMBER = p_is_number;
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
   OPEN c_check_rf1mile (p_is_number,p_org_id);
   FETCH c_check_rf1mile INTO v_rf1mile;
    IF c_check_rf1mile%NOTFOUND THEN
     v_rf1mile := NULL;
    END IF;
   CLOSE c_check_rf1mile;
/* call function to look in storet table for primary/secondary type */
    IF v_rf1mile IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'STA-TR', p_sta_seq);
      IF v_instr_code = 'STA-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_rf1mile IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_sta_seq,NULL);
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_rf1mile := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_orig_item||' rf1mile exists in translation table but has not been mapped to a STORET code. ',NULL,p_sta_seq,NULL);
  END IF;
  RETURN v_rf1mile;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate rf1mile',p_sta_seq,NULL,NULL);
    RETURN NULL;
END validate_rf1mile;
/******************************************************************************************************
    The validate_method_and_datum function takes the original item and checks it against existing
    values in the STORET table TSMMAD. This check is constraint to the appropriate Category (horizontal
    or vertical) and Subcategory (method or datum). If there is a translation the translation is returned and used in a cursor to find the IS number
    org id from TSMMAD.
/****************************************************************************************************/
FUNCTION validate_method_and_datum(p_item_name IN VARCHAR2,p_orig_item IN VARCHAR2,p_sicc_seq IN NUMBER,
                                   p_sta_seq IN NUMBER,p_col_seq IN NUMBER, p_sub_cat IN VARCHAR2,
                                   p_cat IN VARCHAR2) RETURN t_storet_pk_type IS
  v_converted_item           VARCHAR2(80);
  v_tsmmad_pk                t_storet_pk_type;
  v_trans_exist              BOOLEAN := TRUE;
  CURSOR c_check_id_code (p_id_code IN VARCHAR2, p_category IN VARCHAR2,
                          p_sub_category IN VARCHAR2) IS
    SELECT TSMMAD_IS_NUMBER, TSMMAD_ORG_ID FROM TSMMAD WHERE ID_CODE = RPAD(p_id_code,12)
    AND CATEGORY = RPAD(p_category,10) AND SUBCATEGORY = RPAD(p_sub_category,6);
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_sicc_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_id_code(v_converted_item,p_cat,p_sub_cat);
    FETCH c_check_id_code INTO v_tsmmad_pk.is_number, v_tsmmad_pk.org_id;
      IF c_check_id_code%NOTFOUND THEN
        v_tsmmad_pk.is_number := NULL;
        v_tsmmad_pk.org_id := NULL;
      END IF;
    CLOSE c_check_id_code;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_tsmmad_pk.is_number := NULL;
    v_tsmmad_pk.org_id := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||', '||p_item_name||': '||p_orig_item||' exists in translation table but has not been mapped to a STORET code. ',p_sta_seq,NULL,NULL);
  END IF;
  RETURN v_tsmmad_pk;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||', '||p_item_name||': '||SQLERRM||' - in validate_method_and_datum.',p_sta_seq,NULL,NULL);
END validate_method_and_datum;
/******************************************************************************************************
    The validate_aquifer_type function takes the original item and checks it against existing translations.
    If there is a translation the translation is returned and used in a cursor to find the IS number
    org id from TSMGEOUN.
/****************************************************************************************************/
FUNCTION validate_aquifer_type(p_orig_item IN VARCHAR2, p_state_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                                p_item_name IN VARCHAR2, p_col_seq IN NUMBER, p_sta_seq IN NUMBER,
                                p_station_type IN NUMBER)
                                RETURN t_aquifer_type IS
  v_aquifer_type              t_aquifer_type;
  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_choice_id                 VARCHAR2(15);
CURSOR c_check_aquifer_type  (v_converted_item IN VARCHAR2, p_state_code IN VARCHAR2)
       IS SELECT TSMGEOUN_CD,NAME FROM TSMGEOUN
       WHERE NAME = v_converted_item AND STATE_CODE = p_state_code;
BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
   OPEN c_check_aquifer_type (v_converted_item,p_state_code);
   FETCH c_check_aquifer_type INTO v_aquifer_type;
    IF c_check_aquifer_type%NOTFOUND THEN
     v_aquifer_type.tsmgeoun_cd := NULL;
     v_aquifer_type.aquifer_name  := NULL;
    END IF;
   CLOSE c_check_aquifer_type;
/* call function to look in storet table for primary/secondary type */
    IF v_aquifer_type.tsmgeoun_cd IS NULL AND NOT v_trans_exist THEN
     IF p_station_type = 1 THEN -- it's a groundwater station
       v_instr_id  := 'GWS-TR';
       v_choice_id := 'GWS-TR-01';
     ELSE
       v_instr_id  := 'STA-TR';
       v_choice_id := 'STA-TR-01';
     END IF;
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id, p_sta_seq);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_aquifer_type.tsmgeoun_cd IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_sta_seq,NULL);
    END IF;
  ELSIF v_trans_exist THEN
    v_aquifer_type.tsmgeoun_cd := NULL;
    v_aquifer_type.aquifer_name  := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,p_orig_item||' usgs name exists in translation table but has not been mapped to a STORET code. Line '||v_line_cnt,p_sta_seq,NULL,NULL);
  END IF;
--DBMS_OUTPUT.PUT_LINE('at the end of val storet');
  RETURN v_aquifer_type;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate aquifer type',p_sta_seq,NULL,NULL);
    v_aquifer_type.tsmgeoun_cd := NULL;
    v_aquifer_type.aquifer_name  := NULL;
    RETURN v_aquifer_type;
END validate_aquifer_type;
/******************************************************************************************************
    The get_delimiter function checks the table sim_imp_config to find the value entered for the
    delimiter.
/****************************************************************************************************/
FUNCTION get_delimiter(p_imp_cfg_seq IN NUMBER,p_delimiter IN OUT VARCHAR2) RETURN NUMBER IS
  e_no_cfg             EXCEPTION;
  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT SICC_DELIMITER FROM sim_imp_cfg WHERE SICC_SEQ = p_sicc_seq;
BEGIN
  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg INTO p_delimiter;
    IF c_get_imp_cfg%NOTFOUND THEN
      RAISE e_no_cfg;
    END IF;
  CLOSE c_get_imp_cfg;
  RETURN 1;
EXCEPTION
  WHEN e_no_cfg THEN
    RAISE_APPLICATION_ERROR(-20102, 'Configuration invalid or does not exist.');
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20198 , 'Failed to get delimiter.');
    RETURN 0;
END get_delimiter;
/******************************************************************************************************
    The get_item_pos function checks the table sim_imp_config SIM_IMP_CFG_DETS SIM_IMP_COLS to find the value entered for the
    the position of the item in the list
/****************************************************************************************************/
FUNCTION get_item_pos(p_sicc_seq IN NUMBER, p_pos_table IN OUT col_table_num_type,
                      p_default_table IN OUT col_table_type,p_col_cnt IN OUT NUMBER) RETURN NUMBER IS
  v_import_type_seq              NUMBER(10);
  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT SICC_SID_SEQ FROM SIM_IMP_CFG WHERE SICC_SEQ = p_sicc_seq;
  CURSOR c_get_dflt_pos (p_type_seq IN NUMBER) IS
    SELECT SIC_DFLT_POS FROM SIM_IMP_COLS WHERE SIC_SID_SEQ = p_type_seq;
  CURSOR c_get_col_pos (p_sicc_seq IN NUMBER,p_dflt_pos IN NUMBER) IS
    SELECT SICDT_POS, SICDT_DEFAULT FROM SIM_IMP_CFG_DETS, SIM_IMP_COLS
      WHERE SIC_SEQ = SICDT_SIC_SEQ AND SICDT_SICC_SEQ = p_sicc_seq AND SIC_DFLT_POS = p_dflt_pos
      AND SICDT_INCLUDED IS NOT NULL;
BEGIN
  /* get the import type in order to limit the columns selected to the right type */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type INTO v_import_type_seq;
   IF c_get_import_type%NOTFOUND THEN
     v_import_type_seq := NULL;
     RAISE_APPLICATION_ERROR(-20102, 'Unable to get import type from import configuration.');
   END IF;
  CLOSE c_get_import_type;
  p_col_cnt := 0;
  /* loop through import columns and assign actual position to table indexed by
     default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP
    /* get actual position and item default from configuration table */
    OPEN c_get_col_pos(p_sicc_seq,v_dflt_pos.sic_dflt_pos);
    FETCH c_get_col_pos INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
      IF c_get_col_pos%NOTFOUND THEN
        /* if there is nothing in the configuration table for this column, position and default will be null */
        p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
        p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
      ELSE
        p_col_cnt := p_col_cnt + 1;
      END IF;
    CLOSE c_get_col_pos;
  END LOOP;
  Sim2_Import_Pkg.write_error(v_import_log_seq,'num_cols: '||p_col_cnt,NULL,NULL,NULL);
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20103 , 'Failed to get column positions, item names, and column sequences.');
END get_item_pos;
/******************************************************************************************************/
FUNCTION get_column_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) RETURN NUMBER IS
  v_imp_col_seq                   NUMBER(10);
  e_no_cfg_found                  EXCEPTION;
  CURSOR c_get_col_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) IS
    SELECT SICDT_SIC_SEQ FROM sim_imp_cfg, sim_imp_cfg_dets
      WHERE sicdt_sicc_seq = sicc_seq AND SICC_SEQ = p_cfg_seq
      AND sicdt_pos = p_cfg_pos;
BEGIN
  OPEN c_get_col_seq(p_cfg_seq,p_cfg_pos);
  FETCH c_get_col_seq INTO v_imp_col_seq;
    IF c_get_col_seq%NOTFOUND THEN
      v_imp_col_seq := NULL;
      RAISE e_no_cfg_found;
    END IF;
  CLOSE c_get_col_seq;
  RETURN v_imp_col_seq;
EXCEPTION
  WHEN e_no_cfg_found THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - unable to find sequence in sim_imp_cfg_dets for pos: '||p_cfg_pos,NULL,NULL,NULL);
    RAISE_APPLICATION_ERROR(-20100,'Import configuration is not complete');
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - getting column sequence.',NULL,NULL,NULL);
    RETURN NULL;
END get_column_seq;
/******************************************************************************************************/
FUNCTION check_translations(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,
                            p_trans_exist OUT BOOLEAN) RETURN VARCHAR2 IS
  v_storet_value             VARCHAR2(120);
  CURSOR c_get_storet_code(p_code IN VARCHAR2,  p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT st_storet_code FROM SIM_TRANSLATIONS, SIM_IMP_CFG_DETS WHERE st_user_code = p_code
      AND st_sicdt_seq = sicdt_seq AND sicdt_sicc_seq = p_imp_cfg_seq
      AND sicdt_sic_seq = p_col_seq;
BEGIN
  /* check translations table to see if the code should be swapped with another value */
  OPEN c_get_storet_code(p_code,p_imp_cfg_seq,p_col_seq);
  FETCH c_get_storet_code INTO v_storet_value;
    IF c_get_storet_code%NOTFOUND THEN
      /* if not in translations table return original value - will check elsewhere for sequences */
      v_storet_value := p_code;
      p_trans_exist := FALSE;
    ELSIF c_get_storet_code%FOUND THEN
      /* if the code is in the translations table, flag so it's not recreated */
      p_trans_exist := TRUE;
    END IF;
  CLOSE c_get_storet_code;
  RETURN v_storet_value;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check translations',NULL,NULL,NULL);
END check_translations;
/******************************************************************************************************/
PROCEDURE create_trans_record(p_code IN VARCHAR2,p_imp_cfg_seq IN NUMBER,p_col_seq IN NUMBER,
                              p_sta_seq IN NUMBER) IS
  v_st_seq          NUMBER(10);
  v_sicdt_seq       NUMBER(10);
  v_desc            VARCHAR2(80);
  e_imp_cfg_dets    EXCEPTION;
  e_trans_error     EXCEPTION;
  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SICDT_SEQ FROM SIM_IMP_CFG_DETS WHERE SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = p_col_seq;
  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL FROM DUAL;
BEGIN
  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq INTO v_sicdt_seq;
    IF c_get_cfg_det_seq%NOTFOUND THEN
      v_sicdt_seq := NULL;
      RAISE e_imp_cfg_dets;
    END IF;
  CLOSE c_get_cfg_det_seq;
  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq INTO v_st_seq;
    IF c_get_st_seq%NOTFOUND THEN
      v_st_seq := NULL;
      RAISE e_trans_error;
    END IF;
  CLOSE c_get_st_seq;
  INSERT INTO SIM_TRANSLATIONS (
    ST_SEQ,
    ST_SICDT_SEQ,
    ST_USER_CODE,
    ST_STORET_CODE
    )
  VALUES (
    v_st_seq,                  /* st_seq */
    v_sicdt_seq,               /* st_sicdt_seq */
    p_code,                    /* st_user_code */
    NULL                       /* st_storet_code */
    );
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Created translation record for '||p_code,NULL,p_sta_seq,NULL);
EXCEPTION
  WHEN e_imp_cfg_dets THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to find import configuration - stopped import',NULL,p_sta_seq,NULL);
    RAISE_APPLICATION_ERROR(-20101,'Unable to find import configuration');
  WHEN e_trans_error THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to create new record in translations table - error getting sequence',NULL,p_sta_seq,NULL);
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - adding record to translations table.',NULL,p_sta_seq,NULL);
END create_trans_record;
/******************************************************************************************************
    The check allowable values function uses the item input from test and
    SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_CFG_DETS tables are queried to obtain the
    sia_seq
/****************************************************************************************************/
FUNCTION check_allowable_values(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,p_sta_seq IN NUMBER)
  RETURN NUMBER IS
  v_sia_seq         NUMBER(10);
  CURSOR c_check_allowable_val(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SIA_SEQ FROM SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_CFG_DETS
      WHERE SIA_SIC_SEQ = SIC_SEQ AND SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = SIC_SEQ AND SIC_SEQ = p_col_seq AND SIA_ITEM = p_item;
BEGIN
  /* make sure item is an allowable value */
  OPEN c_check_allowable_val(p_item, p_imp_cfg_seq, p_col_seq);
  FETCH c_check_allowable_val INTO v_sia_seq;
    IF c_check_allowable_val%NOTFOUND THEN
      /* if not an allowable value, return null - will write error message, change status of activity */
      v_sia_seq := NULL;
    END IF;
  CLOSE c_check_allowable_val;
  RETURN v_sia_seq;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check allowable values',NULL,p_sta_seq,NULL);
    RETURN NULL;
END check_allowable_values;
/******************************************************************************************************
  function to look in the translations and allowable values tables for items that are
  hard coded values in STORET
********************************************************************************************************/
FUNCTION validate_alval_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                             p_col_seq IN NUMBER, p_sta_seq IN NUMBER, p_station_type IN NUMBER) RETURN NUMBER IS
   v_converted_item           VARCHAR2(80);
   v_item_sia_seq             NUMBER(10);
   v_trans_exist              BOOLEAN := TRUE;
   v_instr_code               VARCHAR2(15);
   v_instr_id                 VARCHAR2(8);
   v_choice_id                VARCHAR2(15);
BEGIN
  /* kms - function to look in the translations and allowable values tables for items that are
     hard coded values in STORET */
  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    /* look in allowable values */
    v_item_sia_seq := check_allowable_values(v_converted_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
    IF v_item_sia_seq IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
-- insert code for gw or stations
    IF p_station_type = 1 THEN -- it's a groundwater station
       v_instr_id  := 'GWS-TR';
       v_choice_id := 'GWS-TR-01';
    ELSE
       v_instr_id  := 'STA-TR';
       v_choice_id := 'STA-TR-01';
    END IF;
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id, p_sta_seq);
        --sim2_import_pkg.write_error(v_import_log_seq, 'instr_cd '||v_instr_code,p_sta_seq,null,null);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
        /* if record could not be matched to anything in translations or allowable values tables, create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value - translation record has been created.',p_sta_seq,NULL,NULL);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value.',p_sta_seq,NULL,NULL);
      END IF;
    ELSIF v_item_sia_seq IS NULL AND v_trans_exist THEN
      /* item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq,NULL,NULL);
    END IF;
  ELSIF v_trans_exist AND v_converted_item IS NULL THEN
    /* if there is a code in the translations table but no value to translate to, write error message
       status will be changed to 'N' later in the program */
    v_item_sia_seq := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' exists in translations table but has not been mapped to a STORET code.',p_sta_seq,NULL,NULL);
  END IF;
--DBMS_OUTPUT.PUT_LINE('at the end of validate al value');
  RETURN v_item_sia_seq;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in validate allowable values.',p_sta_seq,NULL,NULL);
    RETURN NULL;
END validate_alval_item;
/******************************************************************************************************/
FUNCTION write_error_message(p_item_name IN VARCHAR2,p_req IN BOOLEAN,p_line_nbr IN NUMBER,
                             p_provided IN BOOLEAN,p_status IN OUT VARCHAR2, p_comment IN OUT VARCHAR2,
                             p_sta_seq IN NUMBER) RETURN NUMBER IS
BEGIN
  IF p_provided THEN
    p_status := 'N';
--    p_comment := RTRIM(p_item_name||' could not be found or mapped to valid STORET value. '||NVL(p_comment,' '));
--    sim2_import_pkg.write_error(v_import_log_seq, SUBSTR(INITCAP(p_item_name)||' could not be found or mapped to a valid STORET value, line: '||p_line_nbr,1,240),null,p_sta_seq,null);
  ELSE
    IF p_req THEN
      p_status := 'N';
--      p_comment := RTRIM(p_item_name||' not included. '||NVL(p_comment,' '));
      Sim2_Import_Pkg.write_error(v_import_log_seq, SUBSTR('Required item, '||p_item_name||' not provided, line: '||p_line_nbr|| ' activity will not export.',1,240),NULL,p_sta_seq,NULL);
    ELSE
      p_status := p_status;
    END IF;
  END IF;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in write error message.',NULL,p_sta_seq,NULL);
    RETURN 0;
END write_error_message;
/******************************************************************************************************/
FUNCTION get_date(p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,p_cfg_seq IN NUMBER,p_line_cnt IN NUMBER,p_seq IN NUMBER)
  RETURN DATE IS
  v_start_date                  DATE;
  v_fmt                         VARCHAR2(80);
  e_no_cfg_found                EXCEPTION;
  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      v_fmt := NULL;
      /* no format specified, use default */
      v_start_date := TRUNC(TO_DATE(p_raw_date,'MM-DD-RRRR'));
    END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_date := TRUNC(TO_DATE(p_raw_date,v_fmt));
  END IF;
  RETURN v_start_date;
EXCEPTION
  WHEN e_no_cfg_found THEN
    RAISE_APPLICATION_ERROR(-20100,'Import configuration for start or end date is incomplete or does not exist.');
    RETURN NULL;
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' -Construction date  '||p_raw_date||' -format does not match given import format '||v_fmt||' Line: '||p_line_cnt,p_seq,NULL,NULL);
    RETURN NULL;
END get_date;
/******************************************************************************************************/
FUNCTION get_time(p_raw_time IN VARCHAR2,p_col_seq IN NUMBER,p_cfg_seq IN NUMBER)
  RETURN DATE IS
  v_start_time                  DATE;
  v_fmt                         VARCHAR2(80);
  e_no_cfg_found                EXCEPTION;
  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      v_fmt := NULL;
      /* no format specified, use default */
      v_start_time := TO_DATE(SUBSTR(p_raw_time,1,8),'HH24:MI:SS');
--      RAISE e_no_cfg_found;
    END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_time := TO_DATE(SUBSTR(p_raw_time,1,8),v_fmt);
  END IF;
  RETURN v_start_time;
EXCEPTION
  WHEN e_no_cfg_found THEN
    RAISE_APPLICATION_ERROR(-20100,'Import configuration for start or end time is incomplete or does not exist.');
    RETURN NULL;
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in get time.',NULL,NULL,NULL);
    RETURN NULL;
END get_time;
/************************************************************************************************************
  The get_station_is_nbr function finds the station IS number and ORG id from  the Storet
  table TSMSTATN The values are returned in a variable that is TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #
/************************************************************************************************************/
FUNCTION get_station_is_nbr (p_station_id IN VARCHAR2, p_org_id IN VARCHAR2,p_item_name IN VARCHAR2, p_sta_seq IN NUMBER)
  RETURN T_STORET_PK_TYPE IS
  v_station_pk          T_STORET_PK_TYPE;
  CURSOR c_find_station (p_station_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSMSTATN_IS_NUMBER,TSMSTATN_ORG_ID FROM TSMSTATN
    WHERE IDENTIFICATION_CD = RPAD(p_station_id,15) AND TSMSTATN_ORG_ID = RPAD(p_org_id,8);
BEGIN
  OPEN c_find_station (p_station_id ,p_org_id);
  FETCH c_find_station INTO v_station_pk;
    IF c_find_station%NOTFOUND THEN
      v_station_pk.is_number := NULL;
      v_station_pk.org_id := NULL;
    END IF;
  CLOSE c_find_station;
  RETURN v_station_pk;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||
                                 ' - PROC get_station_is_nbr, line: '||v_line_cnt,NULL,p_sta_seq,NULL);
      v_station_pk.is_number := NULL;
      v_station_pk.org_id := NULL;
      RETURN v_station_pk;
END get_station_is_nbr;
/******************************************************************************************************
  The get_tsmvstc_is_nbr function finds the primary/secondary type IS number and ORG id from  the Storet
  table TSMVSTC The values are returned in a variable that is TYPE T_STORET_TSMVSTC_TYPE (HOLDS BOTH THE ORG AND IS #
/************************************************************************************************************/
FUNCTION get_tsmvstc_is_nbr (p_vstc_primary_id IN VARCHAR2, p_vstc_secondary_id IN VARCHAR2, p_item_name IN VARCHAR2)
  RETURN T_STORET_TSMVSTC_TYPE IS
  v_storet_tsmvstc         T_STORET_TSMVSTC_TYPE;
  v_rowcount         NUMBER(10);
  v_is_number        NUMBER(10);
  v_trans_2ndary     VARCHAR2(30);
CURSOR c_count_primary_types  IS
       SELECT COUNT(PRIMARY_TYPE_CD) FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_vstc_primary_id;
CURSOR c_find_vstc (p_vstc_primary_id IN VARCHAR2, p_vstc_secondary_id IN VARCHAR2) IS
       SELECT TSMVSTC_IS_NUMBER,TSMVSTC_ORG_ID FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_vstc_primary_id AND SECONDARY_TYPE_CD = p_vstc_secondary_id;
CURSOR c_find_vstc2 (p_vstc_primary_id IN VARCHAR2) IS
       SELECT TSMVSTC_IS_NUMBER,TSMVSTC_ORG_ID FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_vstc_primary_id;
BEGIN
v_trans_2ndary := TRANSLATE(p_vstc_secondary_id,'1 ', '1'); -- If v_trans_secondary is nothing but spaces then this will be a null
  BEGIN
  OPEN c_count_primary_types;
  FETCH c_count_primary_types INTO v_rowcount;
     IF c_count_primary_types%NOTFOUND THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
     ELSIF v_rowcount > 1 AND v_trans_2ndary IS NULL THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
      v_storet_tsmvstc.tsmvstc_org_id := NULL;
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, p_vstc_primary_id||' -Primary Station Type Needs Secondary, line:'||v_line_cnt,NULL,NULL,NULL);
      v_status := 'N';
      RETURN v_storet_tsmvstc;
     END IF;
  CLOSE c_count_primary_types;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||P_VSTC_PRIMARY_ID||' - count primary type line:'||v_line_cnt,NULL,NULL,NULL);
  END;
 IF p_vstc_secondary_id IS NOT NULL THEN
  BEGIN
  OPEN c_find_vstc (p_vstc_primary_id ,p_vstc_secondary_id);
  FETCH c_find_vstc INTO v_storet_tsmvstc;
    IF c_find_vstc%NOTFOUND THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
    END IF;
  CLOSE c_find_vstc;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' -Error: Cannot find TSMVSTC IS number or Org Id line: '||v_line_cnt,NULL,NULL,NULL);
  END;
 END IF;
IF p_vstc_secondary_id IS NULL THEN
  BEGIN
  OPEN c_find_vstc2 (p_vstc_primary_id);
  FETCH c_find_vstc2 INTO v_storet_tsmvstc;
    IF c_find_vstc2%NOTFOUND THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
    END IF;
  CLOSE c_find_vstc2;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' -Primaty Type error in find vstc2, line:'||v_line_cnt,NULL,NULL,NULL);
  END;
 END IF;
  RETURN v_storet_tsmvstc;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - In Function GET_VSTC_IS_NBR line: '||v_line_cnt,NULL,NULL,NULL);
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
      v_storet_tsmvstc.tsmvstc_org_id := NULL;
      RETURN v_storet_tsmvstc;
END GET_TSMVSTC_IS_NBR;
/***************************************************************************************/
/** Get Primary Type is used to determine if we are importing a well, and therefore if we
/** Need to process the code for aquifer and casing
/***************************************************************************************/
FUNCTION get_primary_type(p_tsmvstc_is_number IN NUMBER,p_tsmvstc_org_id IN VARCHAR2) RETURN NUMBER IS
  v_primary_type     VARCHAR2(30);
  CURSOR c_station_primary_type (p_tsmvstc_is_number IN NUMBER,p_tsmvstc_org_id IN VARCHAR2) IS SELECT
        PRIMARY_TYPE_CD
        FROM TSMVSTC
        WHERE TSMVSTC_IS_NUMBER = p_tsmvstc_is_number AND
              TSMVSTC_ORG_ID    = p_tsmvstc_org_id;
BEGIN
  OPEN c_station_primary_type(p_tsmvstc_is_number,p_tsmvstc_org_id);
  FETCH c_station_primary_type INTO v_primary_type;
  CLOSE c_station_primary_type;
  IF v_primary_type = 'Well' THEN
   RETURN 1;
  ELSE
   RETURN 0;
  END IF;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END;
/************************************************************************************************************
  The get_tsmfhu_is_nbr function finds the hydrologic unit IS number and ORG id from  the Storet
  table TSMFHU The values are returned in a variable that is TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #
/************************************************************************************************************/
FUNCTION get_tsmfhu_is_nbr (p_fhu IN VARCHAR2)
  RETURN T_STORET_PK_TYPE IS
  v_converted_item   VARCHAR2(8);
  v_storet_tsmfhu    T_STORET_PK_TYPE;
  v_rowcount         NUMBER(10);
  v_is_number        NUMBER(10);
CURSOR c_find_fhu (p_fhu IN VARCHAR2) IS
       SELECT TSMFHU_IS_NUMBER,TSMFHU_ORG_ID FROM TSMFHU
       WHERE HYDROLOGIC_UNIT_CD = RPAD(p_fhu,8);
BEGIN
  BEGIN
  OPEN c_find_fhu (p_fhu);
  FETCH c_find_fhu INTO v_storet_tsmfhu;
    IF c_find_fhu%NOTFOUND THEN
      v_storet_tsmfhu.is_number := NULL;
    END IF;
  CLOSE c_find_fhu;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - error finding fhu',NULL,NULL,NULL);
  END;
  RETURN v_storet_tsmfhu;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - Function get_fhu_is_nbr ',NULL,NULL,NULL);
      v_storet_tsmfhu.is_number := NULL;
      v_storet_tsmfhu.org_id := NULL;
      RETURN v_storet_tsmfhu;
END GET_TSMFHU_IS_NBR;
/******* END GET TSMFHU IS NUMBER ******/
/************************************************************************************************************
  The get_tsmgeopa_is_nbr function finds the station IS number and ORG id from  the Storet
  table TSMGEOPA The values are returned in a variable that is TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #
  This function accounts for the 4 possible combinations of state and county ID's,
  1 It's a state postal code and a county name
  2 It's a state FIPS and a county name
  3 It's a state postal code and  county FIPS
  4 It's a  state FIPS and county FIPS
/************************************************************************************************************/
FUNCTION get_tsmgeopa_is_nbr (p_state IN VARCHAR2, p_county IN VARCHAR2)
  RETURN T_STORET_PK_TYPE IS
  v_storet_tsmgeopa    T_STORET_PK_TYPE;
/*** four cursors for the four possible combinations of state and county id's *****/
CURSOR c_find_geopa_1(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMGEOPA
       WHERE STATE_POSTAL_CODE = RPAD(p_state,2) AND COUNTY_NAME = p_county;
CURSOR c_find_geopa_2(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMgeopa
       WHERE STATE_FIPS_CODE = RPAD(p_state,2) AND COUNTY_NAME = p_county;
CURSOR c_find_geopa_3(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMGEOPA
       WHERE STATE_POSTAL_CODE = RPAD(p_state,2) AND FIPS_COUNTY_CODE = RPAD(p_county,3);
CURSOR c_find_geopa_4(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMGEOPA
       WHERE STATE_FIPS_CODE = RPAD(p_state,2) AND FIPS_COUNTY_CODE = RPAD(p_county,3);
BEGIN
  BEGIN
    IF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NOT NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NOT NULL THEN -- it's a state postal and a county NAME
        OPEN c_find_geopa_1 (p_state,p_county);
        FETCH c_find_geopa_1 INTO v_storet_tsmgeopa;
          IF c_find_geopa_1%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_1;
    ELSIF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NOT NULL THEN -- It's a state fips and a county name
        OPEN c_find_geopa_2 (p_state,p_county);
        FETCH c_find_geopa_2 INTO v_storet_tsmgeopa;
          IF c_find_geopa_2%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_2;
    ELSIF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NOT NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NULL THEN -- it's a state postal and  county fips
        OPEN c_find_geopa_3 (p_state,p_county);
        FETCH c_find_geopa_3 INTO v_storet_tsmgeopa;
          IF c_find_geopa_3%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_3;
    ELSIF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NULL THEN -- it's a  state fips and county FIPS
        OPEN c_find_geopa_4 (p_state,p_county);
        FETCH c_find_geopa_4 INTO v_storet_tsmgeopa;
          IF c_find_geopa_4%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_4;
      ELSE
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'get TSMgeopa IS COMBO NOT FOUND: '||v_storet_tsmgeopa.is_number ||'p_state -> '||p_state||' p_county -> '||p_county,NULL,NULL,NULL);
    END IF;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - find geopa',NULL,NULL,NULL);
  END;
  RETURN v_storet_tsmgeopa;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||
                                 ' - PROC GET_geopa_IS_NBR ',NULL,NULL,NULL);
      v_storet_tsmgeopa.is_number := NULL;
      v_storet_tsmgeopa.org_id := NULL;
      RETURN v_storet_tsmgeopa;
END GET_TSMgeopa_IS_NBR;
/****************************************************************************************/
/*** river reach ***/
/************************************************************************************************************
  The get_tsmrrr_is_nbr function finds the station IS number and ORG id from  the Storet
  table TSMRRR The values are returned in a variable that is TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #
  This function
/************************************************************************************************************/
FUNCTION get_tsmrrr_is_nbr (p_huc IN VARCHAR2, p_rf1 IN VARCHAR2)
  RETURN T_STORET_PK_TYPE IS
  v_storet_tsmrrr    T_STORET_PK_TYPE;
CURSOR c_find_tsmrrr(p_huc IN VARCHAR2,p_rf1 IN VARCHAR2) IS
       SELECT TSMRRR_IS_NUMBER,TSMRRR_ORG_ID FROM TSMRRR
       WHERE HYDROLOGIC_UNIT_CD = RPAD(p_huc,8) AND SEGMENT_CODE = RPAD(p_rf1,3);
BEGIN
  BEGIN
        OPEN c_find_tsmrrr (p_huc,p_rf1);
        FETCH c_find_tsmrrr INTO v_storet_tsmrrr;
          IF c_find_tsmrrr%NOTFOUND THEN
           v_storet_tsmrrr.is_number := NULL;
          END IF;
        CLOSE c_find_tsmrrr;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - find tsmrrr_is_number',NULL,NULL,NULL);
  END;
  RETURN v_storet_tsmrrr;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||
                                 ' - PROC GET_TSMRRR_IS_NBR ',NULL,NULL,NULL);
      v_storet_tsmrrr.is_number := NULL;
      v_storet_tsmrrr.org_id := NULL;
      RETURN v_storet_tsmrrr;
END GET_TSMRRR_IS_NBR;
/******************************************************************************************************
  The create_station function inserts known good data into the SIM table SIM_STATIONS
  After the station has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_station (p_station_rec IN sim_stations%ROWTYPE) RETURN NUMBER IS
BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATIONS (
                            STA_SEQ,
                            STA_ID,
                            STA_NAME,
                            STA_DESC,
                            STA_ORG_ID,
                            STA_TSMORGAN_ORG_ID,
                            STA_TSMORGAN_IS_NUMBER,
                            STA_STATUS,
                            STA_ESTAB_DATE,
                            STA_EXPORT_SEQ,
                            STA_IMPORT_SEQ,
                            STA_TSMVSTC_IS_NUMBER,
                            STA_TSMVSTC_ORG_ID
                           )
  VALUES (
    p_station_rec.STA_SEQ,                               /* STA_SEQ            */
    p_station_rec.STA_ID,                                /* STA_AC_SEQ         */
    p_station_rec.STA_NAME,                              /* STA_NUMBER         */
    p_station_rec.STA_DESC,                              /* STA_DESC           */
    p_station_rec.STA_ORG_ID,                            /* STATION ORG ID     */
    p_station_rec.STA_TSMORGAN_ORG_ID,                   /* TSMORGAN_ORG_ID    */
    p_station_rec.STA_TSMORGAN_IS_NUMBER,                /* TSMORGAN_IS_NUMBER */
    p_station_rec.STA_STATUS,                            /* STA_IS_NUMBER      */
    p_station_rec.STA_ESTAB_DATE,                        /* STA_ESTAB_DATE     */
    p_station_rec.STA_EXPORT_SEQ,                        /* STA_EXPORT_SEQ     */
    p_station_rec.STA_IMPORT_SEQ,                        /* STA_IMPORT_SEQ     */
    p_station_rec.STA_TSMVSTC_IS_NUMBER,                 /* STA_IMPORT_SEQ     */
    p_station_rec.STA_TSMVSTC_ORG_ID                     /* STA_SECONDARY_TYPE */
    );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station.',p_station_rec.sta_seq,NULL,NULL);
    RETURN 0;
END create_station;
/******************************************************************************************************
  The create_station_location function inserts known good data into the SIM table SIM_STATIONS
  After the station_location has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_station_location (p_station_location_rec IN sim_station_locations%ROWTYPE) RETURN NUMBER IS
BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATION_LOCATIONS (
                                      STL_TSMRRR_IS_NUMBER,
                                      STL_TSMRRR_ORG_ID,
                                      STL_TSMFHU_ORG_ID,
                                      STL_TSMFHU_IS_NUMBER,
                                      STL_SEQ,
                                      STL_LONG_DIR,
                                      STL_LAT_DIR,
                                      STL_STA_SEQ,
                                      STL_LAT_DEGREES,
                                      STL_LAT_MINUTES,
                                      STL_LAT_SECONDS,
                                      STL_LONG_DEGREES,
                                      STL_LONG_MINUTES,
                                      STL_LONG_SECONDS,
                                      STL_GPS_SCALE,
                                      STL_NRCS,
                                      STL_RF3,
                                      STL_TSMGEOPA_IS_NUMBER,
                                      STL_TSMGEOPA_ORG_ID,
                                      STL_LONG_DEC_MINUTES,
                                      STL_LAT_DEC_MINUTES,
                                      STL_TSMMAD_DATUM,
                                      STL_RF1,
                                      STL_TSMMAD_METHOD,
                                      STL_ELEVATION_MSR,
                                      STL_ELEVATION_UNIT_SEQ,
                                      STL_ELEV_METH_SEQ,
                                      STL_ELEV_DAT_SEQ
                                      )
  VALUES (
         p_station_location_rec.STL_TSMRRR_IS_NUMBER,
         p_station_location_rec.STL_TSMRRR_ORG_ID,
         p_station_location_rec.STL_TSMFHU_ORG_ID,
         p_station_location_rec.STL_TSMFHU_IS_NUMBER,
         p_station_location_rec.STL_SEQ,
         p_station_location_rec.STL_LONG_DIR,
         p_station_location_rec.STL_LAT_DIR,
         p_station_location_rec.STL_STA_SEQ,
         p_station_location_rec.STL_LAT_DEGREES,
         p_station_location_rec.STL_LAT_MINUTES,
         p_station_location_rec.STL_LAT_SECONDS,
         p_station_location_rec.STL_LONG_DEGREES,
         p_station_location_rec.STL_LONG_MINUTES,
         p_station_location_rec.STL_LONG_SECONDS,
         p_station_location_rec.STL_GPS_SCALE,
         p_station_location_rec.STL_NRCS,
         p_station_location_rec.STL_RF3,
         p_station_location_rec.STL_TSMGEOPA_IS_NUMBER,
         p_station_location_rec.STL_TSMGEOPA_ORG_ID,
         p_station_location_rec.STL_LONG_DEC_MINUTES,
         p_station_location_rec.STL_LAT_DEC_MINUTES,
         p_station_location_rec.STL_TSMMAD_DATUM,
         p_station_location_rec.STL_RF1,
         p_station_location_rec.STL_TSMMAD_METHOD,
         p_station_location_rec.STL_ELEVATION_MSR,
         p_station_location_rec.STL_ELEVATION_UNIT_SEQ,
         p_station_location_rec.STL_ELEV_METH_SEQ,
         p_station_location_rec.STL_ELEV_DAT_SEQ
         );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station locations.',p_station_location_rec.stl_seq,NULL,NULL);
    RETURN 0;
END create_station_location;
/******************************************************************************************************
  The create_well_station function inserts known good data into the SIM table SIM_STATIONS
  After the station_location has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/******************************************************************************************************/
FUNCTION create_well_station (p_well_station_rec IN sim_station_wells%ROWTYPE) RETURN NUMBER IS
  BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATION_WELLS (
                                  SSW_SSI_SEQ,
                                  SSW_SEQ,
                                  SSW_TOTAL_DEPTH_UNIT,
                                  SSW_STA_SEQ,
                                  SSW_WELL_NO,
                                  SSW_NAME,
                                  SSW_STATUS_SEQ,
                                  SSW_WELL_USE_SEQ,
                                  SSW_PRIMRY_USE_SEQ,
                                  SSW_USERID_CODE,
                                  SSW_TSMSTATN_ORG_ID,
                                  SSW_TOTAL_DEPTH,
                                  SSW_WELL_CONST_METH,
                                  SSW_DEPTH_OF_CASING,
                                  SSW_IS_NUMBER,
                                  SSW_ORG_ID
                                )
  VALUES (
         p_well_station_rec.SSW_SSI_SEQ,
         p_well_station_rec.SSW_SEQ,
         p_well_station_rec.SSW_TOTAL_DEPTH_UNIT,
         p_well_station_rec.SSW_STA_SEQ,
         p_well_station_rec.SSW_WELL_NO,
         p_well_station_rec.SSW_NAME,
         p_well_station_rec.SSW_STATUS_SEQ,
         p_well_station_rec.SSW_WELL_USE_SEQ,
         p_well_station_rec.SSW_PRIMRY_USE_SEQ,
         p_well_station_rec.SSW_USERID_CODE,
         p_well_station_rec.SSW_TSMSTATN_ORG_ID,
         p_well_station_rec.SSW_TOTAL_DEPTH,
         p_well_station_rec.SSW_WELL_CONST_METH,
         p_well_station_rec.SSW_DEPTH_OF_CASING,
         p_well_station_rec.SSW_IS_NUMBER,
         p_well_station_rec.SSW_ORG_ID
    );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station wells.',p_well_station_rec.ssw_seq,NULL,NULL);
    RETURN 0;
END create_well_station;
/******************************************************************************************************
  The create_well_interval function inserts known good data into the SIM table SIM_STATIONS
  After the well_interval has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/******************************************************************************************************/
FUNCTION create_well_interval (p_well_interval_rec IN sim_station_intvl%ROWTYPE) RETURN NUMBER IS
  BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATION_INTVL (
                              SSI_SEQ,
                              SSI_SSW_SEQ,
                              SSI_TSMGEOUN_CD,
                              SSI_GEOUN_NAME,
                              SSI_INTVL_ORG_ID,
                              SSI_START_DEPTH,
                              SSI_STOP_DEPTH,
                              SSI_DEPTH_UNIT,
                              SSI_SSW_IS_NUMBER,
                              SSI_SSW_ORG_ID
                              )
  VALUES (
         p_well_interval_rec.SSI_SEQ,
         p_well_interval_rec.SSI_SSW_SEQ,
         p_well_interval_rec.SSI_TSMGEOUN_CD,
         p_well_interval_rec.SSI_GEOUN_NAME,
         p_well_interval_rec.SSI_INTVL_ORG_ID,
         p_well_interval_rec.SSI_START_DEPTH,
         p_well_interval_rec.SSI_STOP_DEPTH,
         p_well_interval_rec.SSI_DEPTH_UNIT,
         p_well_interval_rec.SSI_SSW_IS_NUMBER,
         p_well_interval_rec.SSI_SSW_ORG_ID
    );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station intvl.',p_well_interval_rec.ssi_seq,NULL,NULL);
    RETURN 0;
END create_well_interval;
/******************************************************************************************************
  The create_well_casing function inserts known good data into the SIM table SIM_STATIONS
  After the station_location has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/******************************************************************************************************/
FUNCTION create_well_casing (p_well_casing_rec IN sim_station_casing%ROWTYPE) RETURN NUMBER IS
  BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATION_CASING (
                                  SSC_TYPE_CODE,
                                  SSC_THICKNESS_MSR,
                                  SSC_THICKNESS_UNIT_CD,
                                  SSC_INSD_DIAMTR_MSR,
                                  SSC_INSD_DIAMTR_UNT_CD,
                                  SSC_START_DEPTH,
                                  SSC_STOP_DEPTH,
                                  SSC_DEPTH_UNITS,
                                  SSC_D_USERID_CODE,
                                  SSC_LAST_UPD_TS,
                                  SSC_SEQ,
                                  SSC_SSW_SEQ,
                                  TSMINTVL_IS_NUMBER,
                                  TSMINTVL_ORG_ID
                              )
  VALUES (
         p_well_casing_rec.SSC_TYPE_CODE  ,
         p_well_casing_rec.SSC_THICKNESS_MSR ,
         p_well_casing_rec.SSC_THICKNESS_UNIT_CD ,
         p_well_casing_rec.SSC_INSD_DIAMTR_MSR,
         p_well_casing_rec.SSC_INSD_DIAMTR_UNT_CD,
         p_well_casing_rec.SSC_START_DEPTH,
         p_well_casing_rec.SSC_STOP_DEPTH,
         p_well_casing_rec.SSC_DEPTH_UNITS,
         p_well_casing_rec.SSC_D_USERID_CODE,
         p_well_casing_rec.SSC_LAST_UPD_TS,
         p_well_casing_rec.SSC_SEQ ,
         p_well_casing_rec.SSC_SSW_SEQ,
         p_well_casing_rec.TSMINTVL_IS_NUMBER,
         p_well_casing_rec.TSMINTVL_ORG_ID
    );
  COMMIT;
  RETURN 1;
  EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station casing',p_well_casing_rec.ssc_seq,NULL,NULL);
    RETURN 0;
END create_well_casing;
/******************************************************************************************************
  The create_well_igua function inserts known good data into the SIM table SIM_STATIONS
  After the well geologic unit has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/******************************************************************************************************/
FUNCTION create_well_igua (p_well_igua_rec IN sim_station_igua%ROWTYPE) RETURN NUMBER IS
  BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATION_igua (
                                 TSMIGUA_IS_NUMBER,
                                 SIG_OTH_GEO_UNIT_NM,
                                 TSMIGUA_ORG_ID,
                                 SIG_P_AQ_IND_CD,
                                 SIG_W_BEAR_IND_CD,
                                 SIG_USER_ID_CD,
                                 SIG_LAST_UPD_CD,
                                 SIG_TSM_INTVL_IS_NBR,
                                 SIG_TSM_INTVL_ORG_ID,
                                 SIG_GEOUN_CD,
                                 SIG_GEOUN_ORG_ID,
                                 SIG_SEQ,
                                 SIG_STA_SEQ
                               )
  VALUES (
         p_well_igua_rec.TSMIGUA_IS_NUMBER,
         p_well_igua_rec.SIG_OTH_GEO_UNIT_NM,
         p_well_igua_rec.TSMIGUA_ORG_ID,
         p_well_igua_rec.SIG_P_AQ_IND_CD,
         p_well_igua_rec.SIG_W_BEAR_IND_CD,
         p_well_igua_rec.SIG_USER_ID_CD,
         p_well_igua_rec.SIG_LAST_UPD_CD,
         p_well_igua_rec.SIG_TSM_INTVL_IS_NBR,
         p_well_igua_rec.SIG_TSM_INTVL_ORG_ID,
         p_well_igua_rec.SIG_GEOUN_CD,
         p_well_igua_rec.SIG_GEOUN_ORG_ID,
         p_well_igua_rec.SIG_SEQ,
         p_well_igua_rec.SIG_STA_SEQ
    );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station igua',p_well_igua_rec.sig_seq,NULL,NULL);
    RETURN 0;
END create_well_igua;
/******************************************************************************************************
  The import_station procedure parses a text file line by line of data that is separated by
  a delimiter typically a pipe |
  the procedure is activated by filling out and activating the import station dialog box in the SIM2
  application. From there we will assign an ORG ID and procede with opening the text file to extract the
  data
/******************************************************************************************************/
PROCEDURE import_station (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                       p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER) IS
  v_success                              NUMBER(1);
  v_check_import_type                    NUMBER(1);
  v_check_primary_type                   NUMBER(1);
  v_delimiter                            CHAR(1);
  v_InFile                               UTL_FILE.FILE_TYPE;
  v_InLine                               VARCHAR2(1000);
  v_line                                 VARCHAR2(1999);
  v_bad_parse                            BOOLEAN;
  v_item_name                            VARCHAR2(30);
  v_item                                 VARCHAR2(256);
  v_ssc_rec              SIM_STATION_CASING%ROWTYPE;
  v_ssi_rec              SIM_STATION_INTVL%ROWTYPE;
  v_ssw_rec              SIM_STATION_WELLS%ROWTYPE;
  v_stl_rec              SIM_STATION_LOCATIONS%ROWTYPE;
  v_sta_rec              SIM_STATIONS%ROWTYPE;
  v_sta_copy_rec         SIM_STATIONS%ROWTYPE;
  v_sig_rec              SIM_STATION_IGUA%ROWTYPE;
  v_line_table                           Sim2_Import_Pkg.col_table_type;
  v_default_table                        Sim2_Import_Station_Pkg.col_table_type;
  v_pos_table                            Sim2_Import_Station_Pkg.col_table_num_type;
  v_well_no                              NUMBER(10);
  v_well_use                             NUMBER(10);
  v_well_primary_use                     NUMBER(10);
  v_well_status                          NUMBER(10);
  v_total_depth                          NUMBER(4);
  v_elev                                 NUMBER(9,4);
  v_elevation_unit                       NUMBER(10);
  v_elevation_unit_test                  VARCHAR2(2);
  v_primary_name                         VARCHAR2(30);
  v_elevation_method                     NUMBER(10);
  v_elevation_datum                      NUMBER(10);
  v_county                               VARCHAR2(25);
  v_interval_1                           VARCHAR2(18);
  v_interval_2                           VARCHAR2(18);
  v_usgsname                             VARCHAR2(80);
  v_tsmgeoun_cd                          VARCHAR2(12);
  v_casmattype                           VARCHAR2(24);
  v_trans_exist                          BOOLEAN := TRUE;
  v_aquifer_start                        NUMBER(8,3) := NULL;
  v_aquifer_stop                         NUMBER(8,3) := NULL;
  v_casing_start                         NUMBER(8,3);
  v_casing_stop                          NUMBER(8,3);
  v_thickness_msr                        NUMBER(6,3);
  v_india_msr                            NUMBER(5,2);
  v_india_unit                           VARCHAR2(2);
  v_constmeth                            VARCHAR2(35);
  v_primaryaq                            VARCHAR2(1);
  v_waterbear                            VARCHAR2(1);
  v_num_cols                             NUMBER(3);
  v_col_seq                              NUMBER(10);
  v_dflt_pos                             NUMBER(3);
  v_tsmvstc_org_id                       VARCHAR2(8) := NULL;
  v_tsmvstc_is_number                    VARCHAR2(8) := NULL;
  v_tsmorgan_org_id                      VARCHAR2(8);
  v_tsmorgan_is_number                   VARCHAR2(8);
  v_tsmgeopa_org_id                      VARCHAR2(8);
  v_tsmgeopa_is_number                   NUMBER(8);
  v_tsmfhu_org_id                        VARCHAR2(8);
  v_tsmfhu_is_number                     NUMBER(8);
  v_tsmrrr_org_id                        VARCHAR2(8);
  v_tsmrrr_is_number                     NUMBER(8);
  v_primary_checked                      VARCHAR2(1);
  v_proj_is_nbr                          VARCHAR2(8);
  v_name                                 VARCHAR2(60);
  v_type1                                VARCHAR2(30);
  v_estab_date                           DATE;
  v_seq                                  NUMBER(10);
  v_stl_seq                              NUMBER(10);
  v_desc                                 VARCHAR2(1999);
  v_id                                   VARCHAR2(15);
  v_rf3                                  VARCHAR2(12);
  v_nrcs                                 VARCHAR2(8);
  v_sta_desc                             VARCHAR2(1999);
  v_travel_dir                           VARCHAR2(254);
  v_ecoregion                            VARCHAR2(60);
  v_water_depth                          NUMBER(8,3);
  v_depth_unit                           VARCHAR2(2) := 'ft';
  v_casing_depth_unit                    VARCHAR2(2) := 'ft';
  v_aquifer_depth_unit                   VARCHAR2(2) := 'ft';
  v_elevation_msr                        NUMBER(9,4);
  v_lat_degrees                          NUMBER(2);
  v_lat_minutes                          NUMBER(2);
  v_lat_seconds                          NUMBER(6,4);
  v_lat_dir                              VARCHAR2(1) := 'X';
  v_lat_seq								 NUMBER(10);
  v_lat_dec_minutes                      NUMBER(6,4);
  v_long_degrees                         NUMBER(3);
  v_long_minutes                         NUMBER(2);
  v_long_seconds                         NUMBER(6,4);
  v_long_dec_minutes                     NUMBER(8,4);
  v_long_dir                             VARCHAR2(1) := 'X';
  v_long_seq							 NUMBER(10);
  v_datum                                VARCHAR2(120);
  v_meth                                 VARCHAR2(120);
  v_gps_scale                            VARCHAR2(20);
  v_state                                VARCHAR2(2);
  v_mileage                              NUMBER(5,1);
  v_native                               VARCHAR2(5);
  v_rf1                                  VARCHAR2(3);
  v_huc                                  VARCHAR2(8);
  v_comment                              VARCHAR(254);
  v_tsmstatn_is_nbr                      NUMBER(8);
  v_lat_complete                         BOOLEAN;
  v_long_complete                        BOOLEAN;
  v_aquifer_type                         t_aquifer_type;
  v_storet_pk                            t_storet_pk_type;
  v_storet_tsmvstc                       t_storet_tsmvstc_type;
  v_storet_tsmgeopa                      t_storet_pk_type;
  v_storet_tsmfhu                        t_storet_pk_type;
  v_storet_tsmrrr                        t_storet_pk_type;
  v_tsmstatn_org_id                      VARCHAR2(8);
  v_import_desc                          VARCHAR2(200);
  v_comp                                 NUMBER(1);
  v_activity_seq                         NUMBER(10);
  v_depth                                NUMBER(7,2) := NULL;
  v_start_date                           DATE;
  v_start_time                           DATE;
  v_time_zone                            NUMBER(10);
  v_end_date                             DATE;
  v_end_time                             DATE;
  v_secondary_type                       VARCHAR2(30);
  v_dummy                                BOOLEAN;
  v_char_is_nbr                          NUMBER(8);
  v_char_org_id                          VARCHAR2(8);
  v_test_num                             NUMBER(7,4);
  v_test_lat_deg                         NUMBER(2);
  v_test_long_deg                        NUMBER(3);
  v_test_min                             NUMBER(6,4);
  v_test_sec                             NUMBER(6,4);
  v_no_projects                          NUMBER(2);
  v_rows                                 NUMBER(2);
  v_error_cnt                            NUMBER(10);
  v_complete_cnt                         NUMBER(10);
  v_station_cnt                          NUMBER(10);
  v_well_name                            VARCHAR2(35);
  v_dummy_seq                            NUMBER(10);
  e_fileread                             EXCEPTION;
  e_bad_org                              EXCEPTION;
  e_no_cfg                               EXCEPTION;
  e_e_bad_delimiter                      EXCEPTION;
  e_bad_delimiter                        EXCEPTION;
  e_NO_DATA_FOUND                        EXCEPTION;
  e_project_is                           EXCEPTION;
  CURSOR c_get_description (p_is_number IN NUMBER, p_org_id IN VARCHAR2)
            IS SELECT DESCRIPTION FROM TSMMAD WHERE
            TSMMAD_IS_NUMBER = p_is_number AND TSMMAD_ORG_ID = RPAD(p_org_id,8);
  CURSOR c_get_station_seq IS SELECT SIM_STATIONS_SEQ.NEXTVAL FROM DUAL;
  CURSOR c_get_station_location_seq IS SELECT SIM_STATION_LOCATIONS_SEQ.NEXTVAL FROM DUAL;
  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT SIPL_SEQ, SIPL_LINE_TEXT 
      FROM SIM_IMP_LINES 
     WHERE SIPL_SIL_SEQ = p_import_seq
       and sipl_line_text is not null  -- bka  added to eliminate blank lines from the import
      ORDER BY SIPL_LINE_NO;
  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT SIC_PROC_ORDER, SIC_DFLT_POS, SIC_NAME, SIC_SEQ
      FROM SIM_IMP_COLS, SIM_IMP_DEFS, SIM_IMP_CFG
      WHERE SICC_SID_SEQ = SID_SEQ AND SIC_SID_SEQ = SID_SEQ AND SICC_SEQ = p_imp_cfg_seq
      ORDER BY SIC_PROC_ORDER;
  CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(sta_seq) FROM SIM_STATIONS, SIM_STATION_LOCATIONS
      WHERE STL_STA_SEQ = STA_SEQ AND STA_IMPORT_SEQ = p_import_log_seq
        AND STA_STATUS = 'A';
  CURSOR c_count_primary_types  IS                     -- DETERMINES IF THERE IS A SECONDARY TYPE
    SELECT COUNT(PRIMARY_TYPE_CD) FROM TSMVSTC
      WHERE PRIMARY_TYPE_CD = v_type1;
BEGIN
/** Find out import type **/
v_check_import_type := CHECK_IMPORT_TYPE(p_imp_cfg_seq); -- Returns a 1 if it is a Groundwater Station
/* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  v_line_cnt := 0;
  BEGIN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Start at '||TO_CHAR(SYSDATE,'HH24:MI'),NULL,NULL,NULL);
/* get tsmorgan_is_number */
    BEGIN
      v_success := get_org_info(p_imp_cfg_seq,p_org_id,v_tsmorgan_is_number);
      IF v_success != 1 OR v_org_is_nbr IS NULL THEN
        RAISE e_bad_org;
      END IF;
    EXCEPTION
      WHEN e_bad_org THEN
        RAISE_APPLICATION_ERROR(-20109,'Unable to find the specified org id in local version of STORET.');
      WHEN OTHERS THEN
        Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - getting org is number.',NULL,NULL,NULL);
        RAISE NO_DATA_FOUND;
    END;
/*  get the delimiter */
    BEGIN
      v_success := get_delimiter(p_imp_cfg_seq,v_delimiter);
      IF v_success != 1 THEN
        RAISE e_no_cfg;
      END IF;
    EXCEPTION
      WHEN e_no_cfg THEN
        RAISE_APPLICATION_ERROR(-20102,'Error getting delimiter from import configuration.');
      WHEN OTHERS THEN
        Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||'- getting delimiter.',NULL,NULL,NULL);
        RAISE NO_DATA_FOUND;
    END;
/* get actual positions and default information for items being imported */
    BEGIN
      v_success := get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table,v_num_cols);
      IF v_success != 1 THEN
        RAISE e_no_cfg;
      END IF;
    EXCEPTION
      WHEN e_no_cfg THEN
        RAISE_APPLICATION_ERROR(-20102, 'Unable to get item information from the configuration table.');
    END;
/* loop through the lines in the text table */
    FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP
      BEGIN
        /* read one line */
        /* get station is number */
/**** get station sequence ******/
         BEGIN
          OPEN c_get_station_seq;
          FETCH c_get_station_seq INTO v_seq;
          CLOSE c_get_station_seq;
        EXCEPTION
          WHEN OTHERS THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq,' - getting stations sequence.',v_seq,NULL,NULL);
        END;
        v_inline := v_text_line.SIPL_LINE_TEXT;
        v_comment := NULL;
        v_line_cnt := v_line_cnt + 1;
        v_status := 'A';
        v_primary_checked := 'N';
        -- send line to parsing routine:
        BEGIN
          Sim2_Import_Pkg.PARSE_LINE_TABLE(v_inline, v_line_table, v_num_cols, v_delimiter,
                                           v_bad_parse,v_import_log_seq);
          IF v_bad_parse THEN
            RAISE e_e_bad_delimiter;
          END IF;
        EXCEPTION
          WHEN e_e_bad_delimiter THEN
            RAISE e_bad_delimiter;
          WHEN OTHERS THEN
            Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' parsing one line.',NULL,NULL,NULL);
            RAISE e_fileread;
        END;
      END;
          /* loop through all STATION columns */
      FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP
          /* get dflt position, item name, and column sequence */
        v_dflt_pos := v_columns.SIC_DFLT_POS;
        v_item_name := v_columns.SIC_NAME;
        v_col_seq := v_columns.SIC_SEQ;
          /* if column is included on the import, continue */
        IF v_pos_table(v_dflt_pos) IS NOT NULL THEN
          /* get the actual item, or use default if null */
          IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
            v_item := v_line_table(v_pos_table(v_dflt_pos));
          ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL AND v_default_table(v_dflt_pos) IS NOT NULL THEN
            v_item := v_default_table(v_dflt_pos);
          ELSE
            v_item := NULL;
          END IF;
          
          /* assign item to appropriate variables and take care of any other procedural issues that need to be
             taken care of for or with a particular item */
             
          IF v_item_name = 'Station ID' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                v_item := UPPER(v_item); -- Storet only does searches on station id's in upper case DCL
                v_comp := CHECK_IF_STATION_EXISTS(v_item, p_org_id);
                IF v_comp = 1 THEN
                  v_id := v_item;
                ELSE
                 -- Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Station with this Station ID already exists in Storet - station will not export '||v_item||' line: '||v_line_cnt,v_seq,NULL,NULL);
                  v_id := v_item;
                  v_status := 'N';
                END IF;
              ELSE
                v_id := NULL;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - getting station id, line: '||v_line_cnt,v_seq,NULL,NULL);
                v_status := 'N';
            END;
            
          ELSIF v_item_name = 'Station Name' THEN
            BEGIN
              v_name := v_item;
            EXCEPTION
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station name, line: '||v_line_cnt,v_seq,NULL,NULL);
            END;
                  ELSIF v_item_name = 'Primary Type' THEN
            BEGIN
             IF v_item IS NOT NULL THEN
                 v_type1 := validate_primary_type(v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_item_name,v_check_import_type);
                OPEN c_count_primary_types; -- Sees if there is a secondary type required for this primary type.
                 FETCH c_count_primary_types INTO v_rows;
                 IF c_count_primary_types%NOTFOUND THEN
                   v_type1 := NULL;
                 ELSIF v_rows = 1 THEN -- Primary type does not need a secondary type
                             v_storet_tsmvstc := get_tsmvstc_is_nbr(v_type1,NULL,v_item_name);
                    v_check_primary_type := get_primary_type(v_storet_tsmvstc.tsmvstc_is_number,v_storet_tsmvstc.tsmvstc_org_id);
                    v_primary_checked := 'Y';
                 ELSIF v_rows > 1 THEN
                    v_primary_checked := 'N'; -- You will need a secondary Type
                 ELSE
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Invalid primary type, v_type1 = '||v_type1||' line: '||v_line_cnt,v_seq,NULL,NULL);
                    v_type1 := NULL;
                    v_tsmvstc_is_number := NULL;
                    v_tsmvstc_org_id := NULL;
                    v_storet_pk.org_id := NULL;
                    v_primary_checked := 'Y';
                    v_status := 'N';
                 END IF;
                CLOSE c_count_primary_types;
              IF v_type1 IS NOT NULL THEN
                 v_tsmvstc_is_number := v_storet_tsmvstc.tsmvstc_is_number;
                 v_tsmvstc_org_id := v_storet_tsmvstc.tsmvstc_org_id;
                 v_storet_pk.org_id := v_storet_pk.org_id;
              END IF;
           ELSE
              v_tsmvstc_is_number := NULL;
              v_tsmvstc_org_id := NULL;
              v_storet_pk.org_id := NULL;
              v_status := 'N';
              v_primary_checked := 'Y';
              v_type1 := NULL;
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Missing primary type, line: '||v_line_cnt,v_seq,NULL,NULL);
           END IF;
            EXCEPTION
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station type, line: '||v_line_cnt,v_seq,NULL,NULL);
                v_status := 'N';
            END;
          ELSIF v_item_name = 'Secondary Type' AND v_type1 IS NOT NULL THEN
            BEGIN
              IF v_item IS NULL AND v_primary_checked = 'N' THEN
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,v_type1||' - Is missing required Secondary type, line: '||v_line_cnt,v_seq,NULL,NULL);
              END IF;
              v_secondary_type := validate_secondary_type(v_item,v_type1,p_imp_cfg_seq,v_col_seq,v_seq,v_item_name,v_check_import_type);
              IF V_SECONDARY_TYPE IS NULL AND v_primary_checked = 'N' THEN
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,v_type1||' - Has incorrect Secondary type, line: '||v_line_cnt,v_seq,NULL,NULL);
              END IF;
              v_storet_tsmvstc := get_tsmvstc_is_nbr(v_type1,v_secondary_type,v_item_name);
              v_primary_checked := 'Y';
              v_tsmvstc_is_number := v_storet_tsmvstc.tsmvstc_is_number;
              v_tsmvstc_org_id := v_storet_tsmvstc.tsmvstc_org_id;
              v_storet_pk.org_id := v_storet_pk.org_id;
            EXCEPTION
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station secondary type, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_status := 'N';
           END;
         ELSIF v_item_name = 'Latitude Degrees' THEN
          BEGIN
          v_lat_complete := FALSE;
          v_lat_dec_minutes := 0;
          v_lat_minutes := 0;
          v_lat_seconds := 0;
          v_test_num := TO_NUMBER(v_item);
          IF v_test_num >= 0.0000 AND v_test_num <= 90.0000 THEN
              v_lat_degrees := FLOOR(v_test_num);
               v_test_num := v_test_num - v_lat_degrees;
                 IF v_test_num > 0.0000 THEN -- USE DECIMAL PORTION OF MEASUREMENT TO FIND MINUTES AND SECONDS
                    v_lat_dec_minutes := (v_test_num)*60;
                    v_lat_minutes := FLOOR(v_lat_dec_minutes);
                    v_lat_seconds := (v_lat_dec_minutes-v_lat_minutes)*60;
                    v_lat_complete := TRUE;
                 END IF;
          ELSE -- Latitude value is in error
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - lat degs out of range (0-90), line: '||v_line_cnt||' value -> '||v_item,v_seq,NULL,NULL);
          END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_lat_degrees := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station latitude degrees, line: '||v_line_cnt,v_seq,NULL,NULL);
          END;
         ELSIF v_item_name = 'Latitude Minutes'  AND v_lat_complete = FALSE THEN
            BEGIN
               v_test_num := TO_NUMBER(v_item);
               IF v_test_num IS NOT NULL THEN
                IF v_test_num >= 0.00 AND v_test_num <= 60.00 THEN   -- check for valid range
                 v_lat_minutes := FLOOR(v_test_num);
                 v_test_num := v_test_num - v_lat_minutes;
                  IF v_test_num > 0 THEN
                    v_lat_seconds := (v_test_num)*60;
                    v_lat_dec_minutes := v_test_num+v_lat_minutes;
                    v_lat_complete := TRUE;
                  END IF;
                ELSE
                 v_status := 'N';
                 v_lat_minutes := NULL;
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - lat mins out of range (0-60), line: '||v_line_cnt||' value -> '||v_test_num,v_seq,NULL,NULL);
                END IF;
               END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                 v_lat_minutes := NULL;
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station latitude minutes, line: '||v_line_cnt,v_seq,NULL,NULL);
            END;
         ELSIF v_item_name = 'Latitude Seconds' AND v_lat_complete = FALSE THEN
            BEGIN
               v_test_num := TO_NUMBER(v_item);
              IF v_test_num  IS NOT NULL THEN
                IF v_item >= 0 AND v_item <= 60 THEN
                 v_lat_seconds := v_test_num;
                 v_lat_dec_minutes := (v_lat_seconds/60) + v_lat_minutes;
                 v_lat_complete := TRUE;
                ELSE
                 v_status := 'N';
                 v_lat_seconds := NULL;
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - lat secs out of range (0-60), line: '||v_line_cnt||' value -> '||v_item,v_seq,NULL,NULL);
                END IF;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_lat_seconds := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station latitude seconds, line: '||v_line_cnt,v_seq,NULL,NULL);
            END;
         ELSIF v_item_name = 'Latitude Direction' THEN
            BEGIN
            IF v_item IS NULL THEN
              v_lat_dir := 'N';
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' Warning: Latitude Direction value must be N',v_seq,NULL,NULL);
            ELSIF v_item = 'N' /* OR v_item = 'S' */ THEN
              v_lat_dir := v_item;
            ELSE
			  v_lat_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
			  IF v_lat_seq IS NOT NULL THEN
                v_lat_dir := get_allowable_value(v_lat_seq,v_seq,NULL);
			  ELSE
                v_status := 'N';
			    v_lat_dir := 'X';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Latitude Direction value must be N, line: '||v_line_cnt,v_seq,NULL,NULL);
			  END IF;
            END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_lat_dir := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station latitude direction, line: '||v_line_cnt,v_seq,NULL,NULL);
            END;
         ELSIF v_item_name = 'Longitude Degrees' THEN
            BEGIN
          v_long_complete := FALSE;
          v_long_dec_minutes := 0;
          v_long_minutes := 0;
          v_long_seconds := 0;
          v_test_num := TO_NUMBER(v_item);
          IF v_test_num >= 0.0000 AND v_test_num <= 180.0000 THEN
              v_long_degrees := FLOOR(v_test_num);
               v_test_num := v_test_num - v_long_degrees;
                 IF v_test_num > 0.0000 THEN -- USE DECIMAL PORTION OF MEASUREMENT TO FIND MINUTES AND SECONDS
                    v_long_dec_minutes := (v_test_num)*60;
                    v_long_minutes := FLOOR(v_long_dec_minutes);
                    v_long_seconds := (v_long_dec_minutes-v_long_minutes)*60;
                    v_long_complete := TRUE;
                 END IF;
          ELSE -- Longitude value is in error
                v_status := 'N';
                v_long_degrees := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - LONG degs OUT OF RANGE (0-180), line: '||v_line_cnt||' value -> '||v_test_num,v_seq,NULL,NULL);
          END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_long_degrees := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station longitude degrees, line: '||v_line_cnt,v_seq,NULL,NULL);
            END;
         ELSIF v_item_name = 'Longitude Minutes' AND v_long_complete = FALSE THEN
            BEGIN
               v_test_num := TO_NUMBER(v_item);
               IF v_test_num IS NOT NULL THEN
                IF v_test_num >= 0.0000 AND v_test_num <= 60.0000 THEN   -- check for valid range
                 v_long_minutes := FLOOR(v_test_num);
                 v_test_num := v_test_num - v_long_minutes;
                  IF v_test_num > 0 THEN
                    v_long_seconds := (v_test_num)*60;
                    v_long_dec_minutes := v_test_num+v_long_minutes;
                    v_long_complete := TRUE;
                  END IF;
                ELSE
                 v_status := 'N';
                 v_long_minutes := NULL;
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - LONG mins OUT OF RANGE (0-60), line: '||v_line_cnt||' value -> '||v_test_num,v_seq,NULL,NULL);
                END IF;
               END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_long_minutes := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station longitude minutes, line: '||v_line_cnt,v_seq,NULL,NULL);
            END;
         ELSIF v_item_name = 'Longitude Seconds' AND v_long_complete = FALSE THEN
            BEGIN
               v_test_num := TO_NUMBER(v_item);
              IF v_test_num  IS NOT NULL THEN
                IF v_item >= 0 AND v_item <= 60 THEN
                 v_long_seconds := v_test_num;
                 v_long_dec_minutes := (v_long_seconds/60) + v_long_minutes;
                 v_long_complete := TRUE;
                ELSE
                v_long_seconds := NULL;
                 v_status := 'N';
                END IF;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_long_seconds := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station longtitude seconds, line: '||v_line_cnt,v_seq,NULL,NULL);
            END;
        ELSIF v_item_name = 'Longitude Direction' THEN
            BEGIN
            IF v_item IS NULL THEN
              v_long_dir := 'W';
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' Warning: Longitude Direction value must be W',v_seq,NULL,NULL);
            ELSIF v_item = /* 'E' OR v_item = */ 'W' THEN
              v_long_dir := v_item;
            ELSE
			  v_long_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
			  IF v_long_seq IS NOT NULL THEN
                v_long_dir := get_allowable_value(v_long_seq,v_seq,NULL);
			  ELSE
                v_status := 'N';
			    v_long_dir := 'X';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Longitude direction value must be W, line: '||v_line_cnt,v_seq,NULL,NULL);
			  END IF;
            END IF;
            EXCEPTION
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station longitude direction, line: '||v_line_cnt,v_seq,NULL,NULL);
               v_status := 'N';
             END;
        ELSIF v_item_name = 'Geopositioning Method' THEN
            BEGIN
             v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
             v_meth := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
             IF v_meth IS NULL THEN
               Sim2_Import_Pkg.write_error(v_import_log_seq, 'Geopositioning method IS a required field, line: '||v_line_cnt,v_seq,NULL,NULL);
                v_status := 'N';
             END IF;
             EXCEPTION
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting GPS Method line: '||v_line_cnt,v_seq,NULL,NULL);
                v_status := 'N';
             END;
        ELSIF v_item_name = 'Geopositioning Datum' THEN
            BEGIN
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
              v_datum := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
             IF v_datum IS NULL THEN
               Sim2_Import_Pkg.write_error(v_import_log_seq, 'Geopositioning datum IS a required field, line: '||v_line_cnt,v_seq,NULL,NULL);
                v_status := 'N';
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting GPS Datum line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Scale' THEN
            BEGIN
             IF v_item IS NOT NULL THEN
              v_gps_scale := v_item;
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting GPS Scale line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'County' THEN
            BEGIN
              v_item := UPPER(v_item); -- dcl all caps for county name
              v_county := validate_county(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq,v_check_import_type);
              IF v_county IS NULL THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Missing County, line: '||v_line_cnt,v_seq,NULL,NULL);
             v_tsmgeopa_is_number := NULL;
             v_tsmgeopa_org_id := NULL;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting County, line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'State' THEN
            BEGIN
            v_item := UPPER(v_item); -- dcl all caps for state name
            IF v_item IS NULL THEN
              v_tsmgeopa_is_number := NULL;
              v_tsmgeopa_org_id := NULL;
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Missing State, line: '||v_line_cnt,v_seq,NULL,NULL);
              v_status := 'N';
            ELSIF v_county IS NOT NULL THEN
                v_state := validate_state(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq,v_check_import_type);
                v_storet_tsmgeopa := get_tsmgeopa_is_nbr(v_state,v_county);
                v_tsmgeopa_is_number := v_storet_tsmgeopa.is_number;
                v_tsmgeopa_org_id := v_storet_tsmgeopa.org_id;
            ELSE
              /* kms (6/27/01) - county has not been provided, write an error message. */
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Unable TO load State since county has NOT been provided, line: '||v_line_cnt,v_seq,NULL,NULL);
            END IF;
             EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting State line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'HUC' THEN
            BEGIN
              v_huc := validate_huc(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq,v_check_import_type);
              v_storet_tsmfhu := get_tsmfhu_is_nbr(v_huc);
              v_tsmfhu_is_number := v_storet_tsmfhu.is_number;
              v_tsmfhu_org_id    := v_storet_tsmfhu.org_id;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting HUC line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'RF1 Reach' THEN
            BEGIN
            IF v_item IS NOT NULL  AND v_huc IS NOT NULL THEN
             v_rf1 := validate_rf1(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq);
              IF v_rf1 >= 0 THEN
               v_storet_tsmrrr := get_tsmrrr_is_nbr(v_huc,v_rf1);
               v_tsmrrr_is_number := v_storet_tsmrrr.is_number;
               v_tsmrrr_org_id    := v_storet_tsmrrr.org_id;
              ELSE
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - RF1 OR HUC IS NULL: '||v_line_cnt,v_seq,NULL,NULL);
              END IF;
            ELSE
             v_tsmrrr_is_number := NULL;
             v_tsmrrr_org_id    := NULL;
             v_rf1 := NULL;
            END IF;
            EXCEPTION
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting rf1 line: '||v_line_cnt,v_seq,NULL,NULL);
                v_status := 'N';
             END;
        ELSIF v_item_name = 'RF1 Mileage'  THEN
            BEGIN
              IF v_item IS NOT NULL AND v_tsmrrr_is_number IS NOT NULL AND v_tsmrrr_org_id IS NOT NULL THEN
               BEGIN
                v_mileage := validate_rf1mile(v_item,v_tsmrrr_is_number,v_tsmrrr_org_id,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq);
               IF (TRANSLATE(v_item, 'a1234567890.- ','a') IS NOT NULL OR  --contains non_numeric characters
                  TRANSLATE(v_item,'a ','a') IS NULL) THEN                 --contains all spaces
                v_mileage := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Non-numerical Rf1 mileage:  '||v_item,v_seq,NULL,NULL);              
               END IF;
               IF v_mileage >= v_item THEN
                  v_mileage := v_item;
               ELSIF v_mileage < v_item THEN
                v_mileage := NULL;
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Rf1_mileage exceeds maximimum LENGTH FOR the reach, line: '||v_line_cnt,v_seq,NULL,NULL);
               END IF;
               END;
             ELSE
               v_mileage := NULL;
             END IF;
             EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' -problem getting rf1 mileage line: '||v_line_cnt,v_seq,NULL,NULL);
             END;

        ELSIF v_item_name = 'RF3 Reach' THEN
            BEGIN
              v_rf3 := v_item;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting rf3 line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'NRCS Watershed ID' THEN
            BEGIN
             IF v_item IS NOT NULL AND TRANSLATE(v_item, 'a1234567890.- ', 'a') IS NULL THEN
              v_nrcs := v_item;
             ELSE
              v_nrcs := NULL;
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting nrcs line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Station Description' THEN
            BEGIN
             IF v_item IS NOT NULL THEN
              v_sta_desc := v_item;
             ELSE
              v_sta_desc := NULL;
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting station description line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
-- The following data is for Groundwater Stations that are wells
        ELSIF v_item_name = 'Well Name'  AND v_check_import_type = 1 AND v_type1 = 'Well' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
               v_well_name := v_item;
              ELSE
               v_well_name := NULL;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting WELL line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Well Number'  AND v_check_import_type = 1  AND v_type1 = 'Well' THEN
            BEGIN
              v_well_no := 1;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting WELL NUMBER line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Establishment Date'  THEN
            BEGIN
             IF v_item IS NULL THEN
              v_estab_date := NULL;
              --Construction date isn't mandatory so this IS okay
              --sim2_import_pkg.WRITE_ERROR(v_import_log_seq,' - Construction date missing - line: '||v_line_cnt,v_seq,null,null);
             ELSE
              v_estab_date := GET_DATE(v_item,v_col_seq,p_imp_cfg_seq,v_line_cnt,v_seq);
             END IF;
            IF v_estab_date IS NULL AND v_item IS NOT NULL THEN
              --if the date is entered but the format is bad, that's an error
              v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Construction date format does not match - line: '||v_line_cnt,v_seq,NULL,NULL);
            END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR(SQLERRM||' - in ESTAB DATE. Item: '||v_item||' Line: '||TO_CHAR(v_line_cnt),1,240),v_seq,NULL,NULL);
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting Construction date line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Construction Method'  AND v_check_import_type = 1  AND v_type1 = 'Well' THEN
            BEGIN
              v_dummy_seq  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
              v_constmeth := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
            --EXCEPTION
              --WHEN OTHERS THEN
                --construction method is not required in STORET
                --v_status := 'N';
                --sim2_import_pkg.WRITE_ERROR(v_import_log_seq,' - getting Well const meth: '||v_constmeth,v_seq,null,null);
             END;
            ELSIF v_item_name = 'Well Use'  AND v_check_import_type = 1  AND v_type1 = 'Well' THEN
            BEGIN
              v_well_use  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
            IF v_well_use IS NULL THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Missing Well Use line: '||v_line_cnt,v_seq,NULL,NULL);
            END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting Well Use line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Well Status'  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
             v_well_status  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
             IF v_well_status IS NULL THEN
              v_status := 'N';
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting Well status, line: '||v_line_cnt,v_seq,NULL,NULL);
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting Well status: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Depth at Completion'  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
              v_total_depth := v_item;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting Total Depth line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Depth Units'  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
             IF v_total_depth IS NOT NULL THEN
              v_dummy_seq  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
              v_depth_unit := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
              IF v_depth_unit IS NULL THEN -- Top of aquifer above land surface elevation, set top of aquifer = land surface elevation
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - depth at completion depth unit null, assuming ft, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_depth_unit := 'ft';     -- Assume feet as a depth unit
              END IF;
             ELSE
              v_depth_unit := NULL;
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting depth at completion depth unit, line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Interval Type 1'  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                v_interval_1 := 'CASING';
              ELSE
                v_interval_1 := NULL;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting first interval type: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Interval Type 2'  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                v_interval_2 := 'GEOLOGIC UNIT';
              ELSE
                v_interval_2 := NULL;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting first interval type: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Depth Units 2' AND v_check_import_type = 1  AND v_check_primary_type = 1 AND v_interval_2 IS NOT NULL THEN
            BEGIN
              v_dummy_seq  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
              v_aquifer_depth_unit := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
              IF v_aquifer_depth_unit IS NULL THEN -- Top of aquifer above land surface elevation, set top of aquifer = land surface elevation
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - aquifer depth unit null, assuming ft, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_aquifer_depth_unit := 'ft';     -- Assume feet as a depth unit
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting aquifer depth unit, line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Start Depth 2' AND v_check_import_type = 1  AND v_check_primary_type = 1 AND v_interval_2 IS NOT NULL THEN
            BEGIN
            /** The aquifer ranges are given relative to MSL, so we have to make it
            to be relative to the surface elevation at the station. First, we make certain the elevation units and aquifer
            depth units are the same. If they are, we just subtract the aquifer start elevation from the surface elevation
            if the result is less than zero we set the top of the aquifer to be the surface elevation  **/
            IF v_aquifer_depth_unit = v_elevation_unit_test AND v_item IS NOT NULL THEN -- Make sure units are the same before calculation
              v_aquifer_start := v_elevation_msr - v_item;
              IF v_aquifer_start < 0 THEN -- Top of aquifer above land surface elevation, set top of aquifer = land surface elevation
                 v_aquifer_start := 0;
              END IF;
            ELSE  -- The program will convert between meters and feet and insert the correct value
               IF  v_aquifer_depth_unit = 'm' AND v_elevation_unit_test = 'ft' THEN
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning - aquifer depth unit != elevation units program will perform conversion, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_aquifer_start := v_elevation_msr*0.3048 - v_item;
                IF v_aquifer_start < 0 THEN -- Top of aquifer above land surface elevation, set top of aquifer = land surface elevation
                 v_aquifer_start := 0;
                END IF;
               ELSIF  v_aquifer_depth_unit = 'ft' AND v_elevation_unit_test = 'm' THEN
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning - aquifer depth unit != elevation units program will perform conversion, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_aquifer_start := v_elevation_msr*3.2808 - v_item;
                IF v_aquifer_start < 0 THEN -- Top of aquifer above land surface elevation, set top of aquifer = land surface elevation
                 v_aquifer_start := 0;
                END IF;
               ELSE
                 NULL;
               END IF;
            END IF;
            IF v_item IS NULL THEN
                v_aquifer_start := 9999.98; -- set start an stop to an obviously false value so that the aquifer is still placed in Storet
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - aquifer start depth NULL line: '||v_line_cnt,v_seq,NULL,NULL);
           END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting aquifer interval start depth: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Stop Depth 2' AND v_check_import_type = 1  AND v_check_primary_type = 1 AND v_interval_2 IS NOT NULL THEN
            /** The aquifer ranges are given relative to MSL, so we have to make it
            relative to the surface elevation at the station. First, we make certain the elevation units and aquifer
            depth units are the same. If they are, we just subtract the aquifer stop elevation from the surface elevation
            if the result is less than zero we raise an error because the bottom of aquifer must be below the surface elevation
            **/
           BEGIN
           IF v_aquifer_depth_unit = v_elevation_unit_test THEN
             IF v_item IS NOT NULL THEN
               v_aquifer_stop := v_elevation_msr - v_item;
              ELSE
               v_aquifer_stop := NULL;
              END IF;
              IF v_aquifer_stop < 0 OR v_aquifer_stop IS NULL OR (v_aquifer_stop <= v_aquifer_start)  THEN-- Bottom of aquifer is above land surface
                v_aquifer_start := 9999.98; -- set start an stop to an obviously false value so that the aquifer is still placed in Storet
                v_aquifer_stop  := 9999.99;
                --sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Aquifer depths are missing or incorrect, start and stop depths set to 9999.99.',v_seq,NULL,NULL);
              END IF;
            ELSE -- The program will convert between meters and feet and insert the correct value
               IF  v_aquifer_depth_unit = 'm' AND v_elevation_unit_test = 'ft' THEN
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Warning - aquifer depth unit != elevation units program will perform conversion, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_aquifer_stop := v_elevation_msr*0.3048 - v_item;
               ELSIF  v_aquifer_depth_unit = 'ft' AND v_elevation_unit_test = 'm' THEN
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Warning - aquifer depth unit != elevation units program will perform conversion, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_aquifer_stop := v_elevation_msr*3.2808 - v_item;
               ELSE
                 v_aquifer_stop := 9999.99;
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - aquifer stop depth unit is NULL, line: '||v_line_cnt,v_seq,NULL,NULL);
                 -- v_status := 'N';
               END IF;
            END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting aquifer stopdepth type: '||v_line_cnt,v_seq,NULL,NULL);
              END;
        ELSIF v_item_name = 'USGS Name'  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
             IF v_item IS NOT NULL THEN
                v_aquifer_type := validate_aquifer_type(v_item,v_state,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq,v_check_import_type);
                v_usgsname := v_aquifer_type.aquifer_name ;
                v_tsmgeoun_cd := v_aquifer_type.tsmgeoun_cd;
                IF v_usgsname IS NOT NULL AND v_aquifer_stop IS NULL THEN
                   v_aquifer_stop := 9999.99;
                   v_aquifer_start := 9999.98;
                END IF;
                IF v_usgsname IS NULL AND v_interval_2 = 'GEOLOGIC UNIT' THEN -- USGS NAME REQUIRED
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - USGS Name Required with INTERVAL 2 specified as GEOLOGIC UNIT - Line '||v_line_cnt,v_seq,NULL,NULL);
                    v_aquifer_stop := NULL;
                    v_aquifer_start := NULL;
                   v_status := 'N';
               END IF;
                IF v_usgsname IS NOT NULL AND v_interval_2 != 'GEOLOGIC UNIT' THEN -- USGS NAME REQUIRED
                 Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning - USGS Name exists but INTERVAL 2 not specified as GEOLOGIC UNIT - Line '||v_line_cnt,v_seq,NULL,NULL);
                 v_usgsname := NULL;
                  v_aquifer_stop := NULL;
                  v_aquifer_start := NULL;
                END IF;
             ELSE
              v_usgsname := NULL; -- NUll out previous stations assignment of this value
              v_item := NULL;
              v_aquifer_start := NULL;
              v_aquifer_stop  := NULL;
              v_aquifer_depth_unit := NULL;
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - USGSname is NULL Line: '||v_line_cnt,v_seq,NULL,NULL);
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting USGSname: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Casing Material'  AND v_check_import_type = 1  AND v_check_primary_type = 1 AND v_interval_1 IS NOT NULL THEN
            BEGIN
              v_dummy_seq  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
              v_casmattype := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
                --IF v_casmattype IS NULL THEN
                  -- v_status := 'N';
                  --sim2_import_pkg.WRITE_ERROR(v_import_log_seq,' - WARNING: casing material type missing or NULL: '||v_line_cnt,v_seq,null,null);
                --END IF;
             EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting casing material type: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Inside Diameter' AND v_interval_1 IS NOT NULL  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
              v_india_msr := TO_NUMBER(v_item);
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting casing inside diameter: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Diameter Units' AND v_interval_1 IS NOT NULL  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
              v_dummy_seq  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
              v_india_unit := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting casing diameter unit: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Start Depth 1' AND v_interval_1 IS NOT NULL  AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                v_casing_start := TO_NUMBER(v_item);
              ELSE
                v_casing_start := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting casing start depth line: '||v_line_cnt,v_seq,NULL,NULL);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
             END;
        ELSIF v_item_name = 'Stop Depth 1' AND v_interval_1 IS NOT NULL AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN
            BEGIN
             IF v_item IS NOT NULL THEN
                v_casing_stop := v_item;
              ELSE
                v_casing_stop := 9999.99;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' -  casing stop depth is NULL line: '||v_line_cnt,v_seq,NULL,NULL);
              END IF;
           EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting casing stopdepth line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Depth Units 1' AND v_interval_1 IS NOT NULL AND v_check_import_type = 1  AND v_check_primary_type = 1 THEN -- Casing Depth Unit
            BEGIN
              v_dummy_seq  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
              v_casing_depth_unit := get_allowable_value(v_dummy_seq,v_seq,v_item_name);
              IF v_casing_depth_unit IS NULL THEN -- assume feet
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - casing depth unit null, assuming ft, line: '||v_line_cnt,v_seq,NULL,NULL);
                 v_casing_depth_unit := 'ft';     -- Assume feet as a depth unit
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting casing depth unit: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Water Bearing'  AND v_check_import_type = 1  AND v_check_primary_type = 1 AND v_interval_2 IS NOT NULL THEN
            BEGIN
                  IF UPPER(v_item) = 'Y' THEN
                   v_waterbear := 'Y';
                  ELSIF UPPER(v_item) = 'N' THEN
                   v_waterbear := 'N';
                  ELSIF UPPER(v_item) = 'YES' THEN
                   v_waterbear := 'Y';
                  ELSIF UPPER(v_item) = 'NO' THEN
                   v_waterbear := 'N';
                  ELSE
                   Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' Waterbear-> : '||v_waterbear ||'Should be Y or N',v_seq,NULL,NULL);
                  END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - getting WaterBear: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Primary Aquifer'  AND v_check_import_type = 1  AND v_check_primary_type = 1 AND v_interval_2 IS NOT NULL THEN
            BEGIN
            IF UPPER(v_item) = 'Y' THEN
             v_primaryaq := 'Y';
            ELSIF UPPER(v_item) = 'N' THEN
             v_primaryaq := 'N';
            ELSIF UPPER(v_item) = 'YES' THEN
             v_primaryaq := 'Y';
            ELSIF UPPER(v_item) = 'NO' THEN
             v_primaryaq := 'N';
            ELSE
             Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' Primary aq-> : '||v_primaryaq ||'Should be Y or N',v_seq,NULL,NULL);
            END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Getting PrimaryAq: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Elevation' THEN
            BEGIN
             IF (v_item IS NULL OR TRANSLATE(v_item,'a ','a') IS NULL) THEN
                --v_status := 'N';
                v_elevation_msr := NULL;
                --sim2_import_pkg.WRITE_ERROR(v_import_log_seq,' - Elevation is NULL or contains only spaces: '||v_line_cnt,v_seq,null,null);
             ELSIF TRANSLATE(v_item, 'a1234567890.- ', 'a') IS NULL AND TRANSLATE(v_item,'a ','a') IS NOT NULL  THEN -- it's a number and not all spaces
                v_elevation_msr := v_item;
             ELSE
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' Warning: Non Numerical Elevation: '||v_item,v_seq,NULL,NULL);
             END IF;
            EXCEPTION
              WHEN OTHERS THEN
                --v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Getting Elevation: '||v_line_cnt,v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Elevation Units' THEN 
            BEGIN                                  -- sfp, modified 062601
              IF v_item IS NOT NULL AND v_elevation_msr IS NOT NULL THEN
                BEGIN
                v_elevation_unit  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
                v_elevation_unit_test := get_allowable_value(v_elevation_unit,v_seq,v_item_name); -- Used for unit consistency in aquifer depth range
                  IF v_elevation_unit IS NULL THEN
                     v_elevation_unit := NULL;
                     v_status := 'N';
                     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Elevation Units is not valid.',v_seq,NULL,NULL);
                  /* kms - error message has already been written */
                  END IF;
                END;
              ELSIF v_item  IS NOT NULL AND v_elevation_msr IS NULL THEN
                v_elevation_unit := NULL;
                --v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Warning: Elevation Units should not be given when elevation has not been provided.',v_seq,NULL,NULL);                
              ELSIF v_item IS NULL AND v_elevation_msr IS NOT NULL THEN
                v_elevation_unit := NULL;
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Elevation Units is required when elevation has been given.',v_seq,NULL,NULL);
              ELSIF v_item IS NULL AND v_elevation_msr IS NULL THEN
                v_elevation_unit := NULL;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Getting elevation_units, line: '||v_line_cnt,v_seq,NULL,NULL);
             END;


        ELSIF v_item_name = 'Elevation Method' THEN --sfp, this section rewritten 062701
            BEGIN                                   
              IF v_item IS NOT NULL AND v_elevation_msr IS NOT NULL THEN
                v_elevation_method  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
                  IF v_elevation_method IS NULL THEN
                     v_status := 'N';
                     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Elevation Method is not an ALLOWABLE value.',v_seq,NULL,NULL);
                  END IF;
              ELSIF v_item IS NOT NULL AND  v_elevation_msr IS NULL THEN
               v_elevation_method := NULL;
             --  v_status := 'N';
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Warning: Elevation Method should not be given when elevation is missing or contains non-numeric characters.',v_seq,NULL,NULL);
              ELSIF v_item IS NULL AND v_elevation_msr IS NOT NULL THEN
               v_elevation_method := NULL;
               v_status := 'N';
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Elevation Method is required when elevation has been given.',v_seq,NULL,NULL);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||SQLERRM||' - Getting elevation method.',v_seq,NULL,NULL);
             END;
        ELSIF v_item_name = 'Elevation Datum' THEN --sfp, this section rewritten 062701
            BEGIN                                  
              IF v_item IS NOT NULL AND v_elevation_msr IS NOT NULL THEN
                v_elevation_datum  := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_check_import_type);
                 IF v_elevation_datum IS NULL THEN
                    v_status := 'N';
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Elevation Datum is not a valid STORET value.',v_seq,NULL,NULL);
                 END IF;
              ELSIF v_item IS NOT NULL AND v_elevation_msr IS NULL THEN
               v_elevation_datum := NULL;
              -- v_status := 'N';
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Warning: Elevation Datum should not be given when elevation is missing or contains non-numeric characters.',v_seq,NULL,NULL);
              ELSIF v_item IS NULL AND v_elevation_msr IS NOT NULL THEN
               v_elevation_datum := NULL;
               v_status := 'N';
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Elevation Datum is required when elevation has been given.',v_seq,NULL,NULL);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' - Getting elevation datum line: '||v_line_cnt,v_seq,NULL,NULL);
             END;
          END IF;
        ELSE
          /* kms - if elevation is present, units, method and datum for elevation must also be there.
             Write error message only if these columns have not been included in the import, otherwise,
             these columns have already been dealt with. */
           IF v_item_name = 'Elevation Units' AND v_elevation_msr IS NOT NULL THEN
             v_status := 'N';
             Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : elevation units are required when elevation has been provided, this column must be included in the import configuration.',v_seq,NULL,NULL);
           ELSIF v_item_name = 'Elevation Method' AND v_elevation_msr IS NOT NULL THEN
             v_status := 'N';
             Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : elevation method is required when elevation has been provided, this column must be included in the import configuration.',v_seq,NULL,NULL);
           ELSIF v_item_name = 'Elevation Datum' AND v_elevation_msr IS NOT NULL THEN
             v_status := 'N';
             Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : elevation datum is required when elevation has been provided, this column must be included in the import configuration.',v_seq,NULL,NULL);
           END IF;
        END IF;
       END LOOP;
/*** END STATION LOOP ***/
      /* check date/time to make sure if date is there, so is time */
      IF v_start_date IS NOT NULL AND v_start_time IS NULL THEN
         /* if no start time, set to 8 a.m. */
         v_end_time := TO_DATE('0800','HH24MI');
      END IF;
      IF v_end_date IS NOT NULL AND v_end_time IS NULL THEN
        /* if no end time, set to 8 a.m. */
        v_end_time := TO_DATE('0800','HH24MI');
      END IF;
      /* create STATION */
      BEGIN
        /* write data to record and send to create result function */
        /* get result sequence */
         BEGIN
          OPEN c_get_station_location_seq;
          FETCH c_get_station_location_seq INTO v_stl_seq;
          CLOSE c_get_station_location_seq;
        EXCEPTION
          WHEN OTHERS THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq,' - getting station location sequence.',v_seq,NULL,NULL);
        END;
IF v_primary_checked = 'N' THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Error - primary type '||v_type1||' needs secondary type, line: '||v_line_cnt,v_seq,NULL,NULL);
        v_status := 'N';
        v_tsmvstc_is_number              := NULL;
        v_tsmvstc_org_id                 := NULL;
END IF;
        v_sta_rec.STA_SEQ                := v_seq;
        v_sta_rec.STA_ID                 := v_id;
        v_sta_rec.STA_TSMORGAN_ORG_ID    := p_org_id;
        v_sta_rec.STA_ORG_ID             := p_org_id;
        v_sta_rec.STA_STATUS             := v_status;
        v_sta_rec.STA_TSMVSTC_IS_NUMBER  := v_tsmvstc_is_number;
        v_sta_rec.STA_TSMVSTC_ORG_ID     := v_tsmvstc_org_id;
        v_sta_rec.STA_TSMORGAN_IS_NUMBER := v_tsmorgan_is_number;
        v_sta_rec.STA_NAME               := v_name;
        v_sta_rec.STA_IMPORT_SEQ         := v_import_log_seq;
        v_sta_rec.STA_ESTAB_DATE         := v_estab_date;
        v_sta_rec.STA_DESC               := v_sta_desc;
        v_stl_rec.STL_LAT_DEGREES        := v_lat_degrees;
        v_stl_rec.STL_LAT_MINUTES        := v_lat_minutes;
        v_stl_rec.STL_LAT_DEC_MINUTES    := v_lat_dec_minutes;
        v_stl_rec.STL_LAT_SECONDS        := v_lat_seconds;
     IF v_lat_dir = 'X' THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Latitude direction - program will insert N line: '||v_line_cnt,v_seq,NULL,NULL);
        v_lat_dir := 'N';
     END IF;
        v_stl_rec.STL_LAT_DIR            := v_lat_dir;
           v_lat_dir := 'X';
        v_stl_rec.STL_LONG_DEGREES       := v_long_degrees;
        v_stl_rec.STL_LONG_MINUTES       := v_long_minutes;
        v_stl_rec.STL_LONG_DEC_MINUTES   := v_long_dec_minutes;
        v_stl_rec.STL_LONG_SECONDS       := v_long_seconds;
     IF v_long_dir = 'X' THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Longitude direction - program will insert W line: '||v_line_cnt,v_seq,NULL,NULL);
        v_long_dir := 'W';
     END IF;
        v_stl_rec.STL_LONG_DIR           := v_long_dir;
        v_long_dir := 'X';
        v_stl_rec.STL_GPS_SCALE          := v_gps_scale;
        v_gps_scale := NULL;
        v_stl_rec.STL_TSMGEOPA_IS_NUMBER := v_tsmgeopa_is_number;
        v_stl_rec.STL_TSMGEOPA_ORG_ID    := v_tsmgeopa_org_id;
        v_stl_rec.STL_TSMFHU_IS_NUMBER   := v_tsmfhu_is_number;
        v_stl_rec.STL_TSMFHU_ORG_ID      := v_tsmfhu_org_id;
        v_stl_rec.STL_TSMRRR_IS_NUMBER   := v_tsmrrr_is_number;
        v_stl_rec.STL_TSMRRR_ORG_ID      := v_tsmrrr_org_id;
        v_stl_rec.STL_RF1                := v_mileage;
        v_stl_rec.STL_RF3                := v_rf3;
        v_stl_rec.STL_NRCS               := v_nrcs;
        v_stl_rec.STL_SEQ                := v_stl_seq;
        v_stl_rec.STL_STA_SEQ            := v_seq;
        v_stl_rec.STL_TSMMAD_METHOD      := v_meth;
        v_stl_rec.STL_TSMMAD_DATUM       := v_datum;
        v_stl_rec.STL_ELEVATION_MSR      := v_elevation_msr;
        v_stl_rec.STL_ELEVATION_UNIT_SEQ := v_elevation_unit;
        v_stl_rec.STL_ELEV_METH_SEQ      := v_elevation_method;
      --v_stl_rec.STL_ELEV_METH_ORG_ID   := v_elevation_method.org_id;
        v_stl_rec.STL_ELEV_DAT_SEQ       := v_elevation_datum;
      --v_stl_rec.STL_ELEV_DAT_ORG_ID    := v_elevation_datum.org_id;
     IF v_check_primary_type = 1 AND v_check_import_type = 1 THEN --Don't load the data below if station is not a well
        v_ssw_rec.SSW_SSI_SEQ            := v_seq;
        v_ssw_rec.SSW_SEQ                := v_seq;
        v_ssw_rec.SSW_TOTAL_DEPTH_UNIT   := v_depth_unit;
        v_ssw_rec.SSW_STA_SEQ            := v_seq;
        v_ssw_rec.SSW_WELL_NO            := v_well_no;
        v_well_no                        := NULL;
        v_ssw_rec.SSW_NAME               := v_well_name;
        v_ssw_rec.SSW_STATUS_SEQ         := v_well_status;
        v_well_status                    := NULL;
        v_ssw_rec.SSW_WELL_USE_SEQ       := v_well_use;
        v_well_use                       := NULL;
        v_ssw_rec.SSW_PRIMRY_USE_SEQ     := v_well_primary_use;
        v_well_primary_use               := NULL;
        v_ssw_rec.SSW_USERID_CODE        := USER;
        v_ssw_rec.SSW_TSMSTATN_ORG_ID    := p_org_id;
        v_ssw_rec.SSW_TOTAL_DEPTH        := v_total_depth;
        v_total_depth                    := NULL;
        v_ssw_rec.SSW_WELL_CONST_METH    := v_constmeth;
        v_constmeth                      := NULL;
        v_ssw_rec.SSW_DEPTH_OF_CASING    := v_casing_stop;
        v_ssw_rec.SSW_IS_NUMBER          := NULL;
        v_ssw_rec.SSW_ORG_ID             := p_org_id;
        v_ssi_rec.SSI_DEPTH_UNIT         := v_aquifer_depth_unit;
        v_ssi_rec.SSI_TSMGEOUN_CD        := v_tsmgeoun_cd;
        v_ssi_rec.SSI_GEOUN_NAME         := v_usgsname;
        v_ssi_rec.SSI_TYPE_CODE          := v_interval_1;
        v_ssi_rec.SSI_SSW_ORG_ID         := p_org_id;
        v_ssi_rec.SSI_SEQ                := v_seq;
        v_ssi_rec.SSI_SSW_SEQ            := v_seq;
        v_ssi_rec.SSI_START_DEPTH        := v_aquifer_start;
        v_ssi_rec.SSI_STOP_DEPTH         := v_aquifer_stop;
        v_ssi_rec.SSI_GEOUN_ORG_ID       := p_org_id;
        v_ssi_rec.SSI_SSW_ORG_ID         := p_org_id;
        v_ssc_rec.SSC_TYPE_CODE          := v_casmattype;
        v_casmattype                     := NULL;
        v_ssc_rec.SSC_START_DEPTH        := v_casing_start;
        v_casing_start := NULL;
        v_ssc_rec.SSC_STOP_DEPTH         := v_casing_stop;
        v_casing_stop := NULL;
        v_ssc_rec.SSC_DEPTH_UNITS        := v_casing_depth_unit;
        v_casing_depth_unit := NULL;
        v_ssc_rec.SSC_THICKNESS_UNIT_CD  := 'in';
        v_ssc_rec.SSC_THICKNESS_MSR      := 0;
        v_ssc_rec.SSC_INSD_DIAMTR_MSR    := v_india_msr;
        v_india_msr := NULL;
        v_ssc_rec.SSC_INSD_DIAMTR_UNT_CD := v_india_unit;
        v_india_unit := NULL;
        v_ssc_rec.SSC_D_USERID_CODE      := USER;
        v_ssc_rec.SSC_LAST_UPD_TS        := SYSDATE;
        v_ssc_rec.SSC_SEQ                := v_seq;
        v_ssc_rec.SSC_SSW_SEQ            := v_seq;
        v_ssc_rec.TSMINTVL_ORG_ID        := p_org_id;
        v_ssc_rec.TSMINTVL_IS_NUMBER     :=  0;
        v_sig_rec.TSMIGUA_IS_NUMBER      :=  0;
        v_sig_rec.SIG_OTH_GEO_UNIT_NM    :=  NULL;
        v_sig_rec.TSMIGUA_ORG_ID         :=  NULL;
        v_sig_rec.SIG_P_AQ_IND_CD        :=  v_primaryaq;
        v_primaryaq                      := NULL;
        v_sig_rec.SIG_W_BEAR_IND_CD      :=  v_waterbear;
        v_waterbear                      := NULL;
        v_sig_rec.SIG_USER_ID_CD         :=  USER;
        v_sig_rec.SIG_LAST_UPD_CD        :=  SYSDATE;
        v_sig_rec.SIG_TSM_INTVL_IS_NBR   :=  NULL;
        v_sig_rec.SIG_TSM_INTVL_ORG_ID   :=  p_org_id;
        v_sig_rec.SIG_GEOUN_CD           :=  v_tsmgeoun_cd;
        v_tsmgeoun_cd                    := NULL;
        v_sig_rec.SIG_GEOUN_ORG_ID       :=  'ZZZZZZZZ';
        v_sig_rec.SIG_SEQ                :=  v_seq;
        v_sig_rec.SIG_STA_SEQ            :=  v_seq;
END IF;
        /* insert station */
        IF v_id IS NOT NULL THEN
           v_success := create_station(v_sta_rec);
        END IF;
        /* if insert function failed, write error and prevent activity from being exported */
        IF v_success != 1 THEN
          UPDATE sim_stations
            SET sta_status = 'N'
          WHERE sta_seq = v_seq;
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Failed to create station in sim_stations, activity will not be exported, line: '||v_line_cnt,v_seq,NULL,NULL);
        END IF;
        IF v_id IS NOT NULL THEN
         v_success := create_station_location(v_stl_rec);
        END IF;
        /* if insert function failed, write error and prevent activity from being exported */
        IF v_success != 1 THEN
          UPDATE sim_stations
            SET sta_status = 'N'
          WHERE sta_seq = v_seq;
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Failed to create station in sim_station_locations, station will not be exported, line: '||v_line_cnt,v_seq,NULL,NULL);
        END IF;
--IF v_check_primary_type = 1 AND v_check_import_type = 1 AND v_primary_name = 'Well' THEN -- only do wells if the type is groundwater and a well
IF v_check_import_type = 1 AND v_type1 = 'Well' THEN -- only do wells if the type is groundwater and a well
        IF v_id IS NOT NULL THEN
         v_success := create_well_station(v_ssw_rec);
        END IF;
        /* if insert function failed, write error and prevent activity from being exported */
        IF v_success != 1 THEN
          UPDATE sim_stations
            SET sta_status = 'N'
          WHERE sta_seq = v_seq;
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Failed to create well in sim_station_wells, station will not be exported, line: '||v_line_cnt,v_seq,NULL,NULL);
        END IF;
        IF v_id IS NOT NULL THEN
         v_success := create_well_interval(v_ssi_rec);
        END IF;
        /* if insert function failed, write error and prevent activity from being exported */
        IF v_success != 1 THEN
          UPDATE sim_stations
            SET sta_status = 'N'
          WHERE sta_seq = v_seq;
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Failed to create well interval sim_station_interval, station will not be exported, line: '||v_line_cnt,v_seq,NULL,NULL);
        END IF;
        IF v_id IS NOT NULL THEN
         v_success := create_well_casing(v_ssc_rec);
        END IF;
        /* if insert function failed, write error and prevent activity from being exported */
        IF v_success != 1 THEN
          UPDATE sim_stations
            SET sta_status = 'N'
          WHERE sta_seq = v_seq;
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Failed to create well casing in sim_station_casing, station will not be exported, line: '||v_line_cnt,v_seq,NULL,NULL);
        END IF;
        IF v_id IS NOT NULL THEN
         v_success := create_well_igua(v_sig_rec);
        END IF;
        /* if insert function failed, write error and prevent activity from being exported */
        IF v_success != 1 THEN
          UPDATE sim_stations
            SET sta_status = 'N'
          WHERE sta_seq = v_seq;
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Failed to create well igua in sim_station_casing, station will not be exported, line: '||v_line_cnt,v_seq,NULL,NULL);
        END IF;
END IF;
      EXCEPTION
        WHEN OTHERS THEN
          v_import_desc := SQLERRM||' - writing a result. Line: '||v_line_cnt;
          Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, v_import_desc,v_seq,NULL,NULL);
      END;
      /* update import lines table with station sequence */
      UPDATE SIM_IMP_LINES SET SIPL_STA_SEQ = v_seq
        WHERE SIPL_SEQ = v_text_line.SIPL_SEQ;
    END LOOP; /* end of text loop */
    OPEN c_comp_count(v_import_log_seq);
    FETCH c_comp_count INTO v_complete_cnt;
    CLOSE c_comp_count;
    v_error_cnt := v_line_cnt - v_complete_cnt;
    /* update counts in the import log */
    UPDATE SIM_IMPORT_LOG
       SET SIL_ROWS_IMPORTED = v_line_cnt,
           SIL_ROWS_ERRORS = v_error_cnt,
           SIL_ROWS_READY = v_complete_cnt
       WHERE SIL_SEQ = v_import_log_seq;
    COMMIT;
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_station_cnt)||' stations loaded',v_seq,NULL,NULL);
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq ,'Finished at '||TO_CHAR(SYSDATE,'HH24:MI'),v_seq,NULL,NULL);
  END;
EXCEPTION
   WHEN e_no_cfg THEN
     RAISE_APPLICATION_ERROR(-20102, 'Configuration invalid or does not exist.');
   WHEN e_bad_delimiter THEN
     RAISE_APPLICATION_ERROR(-20197,'The import file contains the wrong number of columns on line: '||v_line_cnt);
   WHEN OTHERS THEN
       Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR(SQLERRM||' - in sim2_import_result_pkg.import_results. Line: '||TO_CHAR(v_line_cnt),1,240),v_seq,NULL,NULL);
       Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_station_cnt)||' stations loaded',v_seq,NULL,NULL);
       Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,' Finished at '||TO_CHAR(SYSDATE,'HH24:MI'),v_seq,NULL,NULL);
END;
/******************************************************************************************************/
END Sim2_Import_Station_Pkg;
/
